# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2006-2007 Ali Sabil <ali.sabil@gmail.com>
# Copyright (C) 2007 Johann Prieur <johann.prieur@gmail.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import weakref
import logging

import dbus
import telepathy
import hashlib
import vobject

from telepathy.interfaces import CONNECTION

from tpufa.aliasing import UfAAliasing
from tpufa.capabilities import UfACapabilities
from tpufa.handle import UfAHandleFactory
from tpufa.contacts import UfAContacts
from tpufa.channel_manager import UfAChannelManager
from tpufa.avatars import UfAAvatars
from tpufa.android.androidcall import AndroidCall
from tpufa.android.androidmessages import AndroidMessages
from tpufa.call import UfAPhone, UfACall, UfACallConnection

CANONICAL_IFACE_TELEPHONY = "com.canonical.Telephony"

__all__ = ['UfAConnection']

logger = logging.getLogger('UfA.Connection')


class UfAConnection(telepathy.server.Connection,
        telepathy.server.ConnectionInterfaceRequests,
        UfAAliasing,
        UfAAvatars,
        UfACapabilities,
        UfAContacts,
        AndroidCall,
        AndroidMessages):

    phone = None
    createdChannels = []
    # only notify channels after first run
    notifyChannels = False

    contactInfo = {}

    def __init__(self, protocol, manager, parameters):
        protocol.check_parameters(parameters)

        try:
            self._account = "ufa" #unicode(parameters['account'])
            self._channel_manager = UfAChannelManager(self, protocol)

            # Call parent initializers
            telepathy.server.Connection.__init__(self, 'ufa', self._account,
                    'ufa', protocol)
            telepathy.server.ConnectionInterfaceRequests.__init__(self)
            UfAAliasing.__init__(self)
            UfAAvatars.__init__(self)
            UfACapabilities.__init__(self)
            UfAContacts.__init__(self)
            AndroidCall.__init__(self)
            AndroidMessages.__init__(self)

            self_handle = self.create_handle(telepathy.HANDLE_TYPE_CONTACT,
                    self._account)
            self.set_self_handle(self_handle)

            self._implement_property_get(telepathy.CONNECTION, {
                'HasImmortalHandles': lambda: dbus.Boolean(True),
            })

            logger.info("Connection to the account %s created" % self._account)
        except Exception, e:
            import traceback
            logger.exception("Failed to create Connection")
            raise

    def AndroidConnected(self):
        self.notifyChannels = False
        self.createdChannels = []
        # refresh the call state
        defaultPhone = self.getDefaultPhone()
        self.AndroidCallStateChanged(defaultPhone)
        for channel in self.createdChannels:
            # update status of the new channel
            channel.AndroidCallStateChanged(defaultPhone)
            self.signal_new_channels([channel])
            # notify the clients only after channel status update
        self.notifyChannels = True
        self.createdChannels = []

        # create text channels for unread messages
        for message in self.getUnreadMessages():
            self.messageReceived(message, scrollback=True)

    def StatusChanged(self, status, reason):
        telepathy.server.Connection.StatusChanged(self, status, reason)
        if status == telepathy.CONNECTION_STATUS_CONNECTED:
            self.AndroidConnected()

    @property
    def manager(self):
        return self._manager

    def handle(self, handle_type, handle_id):
        self.check_handle(handle_type, handle_id)
        return self._handles[handle_type, handle_id]

    def create_handle(self, handle_type, handle_name, **kwargs):
        """Create new handle with given type and name."""
        handle_id = self.get_handle_id()
        handle = UfAHandleFactory(self, handle_type, handle_id,
                handle_name, **kwargs)
        if handle is None:
            raise telepathy.NotAvailable('Handle type unsupported %d' % handle_type)
        logger.info("New Handle %s" % unicode(handle))
        self._handles[handle_type, handle_id] = handle

        # if the handle is a contact handle, update the capabilities
        if handle_type == telepathy.HANDLE_TYPE_CONTACT:
            self._add_default_capabilities([handle])
            self._update_contact_capabilities([handle])

        return handle

    def is_valid_handle_name(self, handle_type, handle_name):
        """Make sure the name is valid for this type of handle."""
        return True

    def normalize_handle_name(self, handle_type, handle_name):
        """Normalize handle name so the name is consistent everywhere."""
        if not self.is_valid_handle_name(handle_type, handle_name):
            raise telepathy.InvalidHandle('TargetID %s not valid for type %d' %
                (name, handle_type))
        if handle_type == telepathy.HANDLE_TYPE_CONTACT:
            return handle_name.lower().strip()
        return handle_name

    def ensure_contact_handle(self, contact):
        """Build handle name for contact and ensure handle."""
        if contact == "":
            return telepathy.NoneHandler()
        # return old handle if we have it already.
        for handle in self._handles:
            old_handle = self.handle(telepathy.HANDLE_TYPE_CONTACT, handle[1])
            if self.comparePhoneNumbers(old_handle.name, contact):
                return old_handle
        handle_type = telepathy.HANDLE_TYPE_CONTACT
        handle_name = contact
        return self.ensure_handle(handle_type, handle_name, contact=contact)

    def fill_contact_info(self, handle_id):
        # do not cache self handle information
        if handle_id == self.self_handle.id:
            return

        # do not update cache if we have to contact already
        if handle_id in self.contactInfo:
            return

        handle = self.handle(telepathy.HANDLE_TYPE_CONTACT, handle_id)
        # keep only 5 items in memory
        if len(self.contactInfo) == 5:
            self.contactInfo.popitem()

        vcard = self.getContactForNumber(handle.name)
        if len(vcard) == 0:
            return
        parsedVcard = vobject.readOne(vcard, ignoreUnreadable=True)
        alias = None
        if hasattr(parsedVcard, 'fn') and len(parsedVcard.fn.value) != 0:
            alias = parsedVcard.fn.value
        else:
            alias = handle.name
        
        self.contactInfo[handle_id] = {}
        self.contactInfo[handle_id]['alias'] = alias

        avatarToken = None
        if hasattr(parsedVcard, 'photo') and len(parsedVcard.photo.value) != 0:
            avatarToken = hashlib.sha1()
            avatarToken.update(parsedVcard.photo.value)
            self.contactInfo[handle_id]['avatar'] = [ avatarToken.hexdigest(), parsedVcard.photo.value, parsedVcard.photo.type_param.lower() ]

    def Connect(self):
        if self._status == telepathy.CONNECTION_STATUS_DISCONNECTED:
            logger.info("Connecting")
            self.StatusChanged(telepathy.CONNECTION_STATUS_CONNECTING, telepathy.CONNECTION_STATUS_REASON_REQUESTED)
            self.StatusChanged(telepathy.CONNECTION_STATUS_CONNECTED, telepathy.CONNECTION_STATUS_REASON_REQUESTED)

    def Disconnect(self):
        logger.info("Disconnecting")
        self._disconnected()

    def _disconnected(self):
        logger.info("Disconnected")
        self.StatusChanged(telepathy.CONNECTION_STATUS_DISCONNECTED,
                           telepathy.CONNECTION_STATUS_REASON_REQUESTED)
        self._channel_manager.close()
        self._manager.disconnected(self)

    def GetInterfaces(self):
        return self._interfaces

    def _generate_props(self, channel_type, handle, suppress_handler, initiator_handle=None):
        props = {
            telepathy.CHANNEL_INTERFACE + '.ChannelType': channel_type,
            telepathy.CHANNEL_INTERFACE + '.TargetHandle': handle.get_id(),
            telepathy.CHANNEL_INTERFACE + '.TargetHandleType': handle.get_type(),
            telepathy.CHANNEL_INTERFACE + '.Requested': suppress_handler
            }

        if initiator_handle is not None:
            if initiator_handle.get_type() is not telepathy.HANDLE_TYPE_NONE:
                props[telepathy.CHANNEL_INTERFACE + '.InitiatorHandle'] = \
                        initiator_handle.get_id()

        return props


    @dbus.service.method(telepathy.CONNECTION, in_signature='suub',
        out_signature='o', async_callbacks=('_success', '_error'))
    def RequestChannel(self, type, handle_type, handle_id, suppress_handler,
            _success, _error):
        self.check_connected()
        channel_manager = self._channel_manager

        if handle_id == telepathy.HANDLE_TYPE_NONE:
            handle = telepathy.server.handle.NoneHandle()
        else:
            handle = self.handle(handle_type, handle_id)
        props = self._generate_props(type, handle, suppress_handler)
        self._validate_handle(props)

        channel = channel_manager.channel_for_props(props, signal=False)

        _success(channel._object_path)
        self.signal_new_channels([channel])

    def create_call_channels(self, conns):
        for conn in conns:
            # do not create channels for disconnected connections
            if conn.disconnectCause != self.DISCONNECT_CAUSE_NOT_CONNECTED:
                continue
            phoneNumber = conn.address

            if self._channel_manager.call_channel_for_connid(conn.id):
                continue

            handle = None
            if phoneNumber != "":
                handle = self.ensure_contact_handle(phoneNumber)
            else:
                # unknown number
                handle = self.ensure_contact_handle("#")

            props = None
            if conn.isIncoming:
                props = self._generate_props(telepathy.CHANNEL_TYPE_CALL,
                        handle, False, initiator_handle=handle)
            else:
                props = self._generate_props(telepathy.CHANNEL_TYPE_CALL,
                        handle, False, initiator_handle=self.self_handle)

            # check if this channel was already created
            if self._channel_manager.existing_channel(props):
                continue

            props[telepathy.CHANNEL_TYPE_CALL + ".InitialAudio"] = True
            channel = self._channel_manager.channel_for_props(props, signal=self.notifyChannels)
            channel._androidCallId = conn.id
            self.createdChannels.append(channel)

    ### Android Call Interface ###
    def AndroidCallStateChanged(self, calls):
        self.phone = UfAPhone(calls)
        # verify if any call was started from the phone
        if (len(self.phone.foregroundCall.connections) != 0):
            # do not create channels while dialing. it might
            # duplicate channels, since the callid might change
            if self.phone.foregroundCall.state != self.CALL_STATE_DIALING:
                self.create_call_channels(self.phone.foregroundCall.connections)
        if (len(self.phone.backgroundCall.connections) != 0):
            self.create_call_channels(self.phone.backgroundCall.connections)
        if len(self.phone.ringingCall.connections) == 1:
            self.create_call_channels(self.phone.ringingCall.connections)

    ####### Android Messages interface ######
    def messageReceived(self, message, scrollback=False):
        number = message[0]
        text = message[1]
        timestamp = int(message[2])/1000
        id = int(message[6])
        handle = self.ensure_contact_handle(number)
        props = self._generate_props(telepathy.CHANNEL_TYPE_TEXT,
                handle, False)
        channel = self._channel_manager.channel_for_props(props,
                signal=True)
        channel._signal_text_received(handle, text, id, timestamp, scrollback)

    @dbus.service.signal(dbus_interface=CANONICAL_IFACE_TELEPHONY,
                         signature='i')
    def VoicemailCountChanged(self, count):
        pass

    @dbus.service.signal(dbus_interface=CANONICAL_IFACE_TELEPHONY,
                         signature='b')
    def VoicemailIndicatorChanged(self, active):
        pass

    @dbus.service.method(dbus_interface=CANONICAL_IFACE_TELEPHONY,
                         in_signature='', out_signature='b')
    def VoicemailIndicator(self):
        return AndroidCall.getVoiceMessageIndicator(self)

    @dbus.service.method(dbus_interface=CANONICAL_IFACE_TELEPHONY,
                         in_signature='', out_signature='s')
    def VoicemailNumber(self):
        return AndroidCall.getVoiceMessageNumber(self)

    @dbus.service.method(dbus_interface=CANONICAL_IFACE_TELEPHONY,
                         in_signature='', out_signature='i')
    def VoicemailCount(self):
        return AndroidCall.getVoiceMessageCount(self)

