# telepathy-butterfly - an MSN connection manager for Telepathy
#
# Copyright (C) 2009 Collabora Ltd.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import logging
import dbus

import telepathy

__all__ = ['UfACapabilities']

logger = logging.getLogger('UfA.Capabilities')

class UfACapabilities(
        telepathy.server.ConnectionInterfaceCapabilities,
        telepathy.server.ConnectionInterfaceContactCapabilities):

    text_chat_class = \
        ({telepathy.CHANNEL_INTERFACE + '.ChannelType':
              telepathy.CHANNEL_TYPE_TEXT,
          telepathy.CHANNEL_INTERFACE + '.TargetHandleType':
              dbus.UInt32(telepathy.HANDLE_TYPE_CONTACT)},
         [telepathy.CHANNEL_INTERFACE + '.TargetHandle',
          telepathy.CHANNEL_INTERFACE + '.TargetID'])

    audio_chat_class = \
        ({telepathy.CHANNEL_INTERFACE + '.ChannelType':
              telepathy.CHANNEL_TYPE_CALL,
          telepathy.CHANNEL_INTERFACE + '.TargetHandleType':
              dbus.UInt32(telepathy.HANDLE_TYPE_CONTACT)},
         [telepathy.CHANNEL_INTERFACE + '.TargetHandle',
          telepathy.CHANNEL_INTERFACE + '.TargetID',
          telepathy.CHANNEL_TYPE_CALL + '.InitialAudio'])

    def __init__(self):
        telepathy.server.ConnectionInterfaceCapabilities.__init__(self)
        telepathy.server.ConnectionInterfaceContactCapabilities.__init__(self)

        self._update_capabilities_calls = []


    ### Capabilities interface -----------------------------------------------

    def _get_capabilities(self, contact):
        gen_caps = 0
        spec_caps = 0

        gen_caps |= telepathy.CONNECTION_CAPABILITY_FLAG_CREATE
        gen_caps |= telepathy.CONNECTION_CAPABILITY_FLAG_INVITE
        spec_caps |= telepathy.CHANNEL_MEDIA_CAPABILITY_AUDIO
        spec_caps |= telepathy.CHANNEL_MEDIA_CAPABILITY_NAT_TRAVERSAL_STUN

        return gen_caps, spec_caps

    def _add_default_capabilities(self, handles):
        """Add the default capabilities to these contacts."""
        ret = []
        for handle in handles:
            new_flag = telepathy.CONNECTION_CAPABILITY_FLAG_CREATE

            ctype = telepathy.CHANNEL_TYPE_TEXT
            diff = self._diff_capabilities(handle, ctype, added_gen=new_flag)
            ret.append(diff)

            ctype = telepathy.CHANNEL_TYPE_CALL
            diff = self._diff_capabilities(handle, ctype, added_gen=new_flag)
            ret.append(diff)

        self.CapabilitiesChanged(ret)

    def _update_capabilities(self, handle):
        ctype = telepathy.CHANNEL_TYPE_CALL

        new_gen, new_spec = self._get_capabilities(handle.contact)
        diff = self._diff_capabilities(handle, ctype, new_gen, new_spec)
        if diff is not None:
            self.CapabilitiesChanged([diff])


    ### ContactCapabilities interface ----------------------------------------

    def AdvertiseCapabilities(self, add, remove):
        return telepathy.server.ConnectionInterfaceCapabilities.\
            AdvertiseCapabilities(self, add, remove)

    def UpdateCapabilities(self, caps):
        if self._status != telepathy.CONNECTION_STATUS_CONNECTED:
            self._update_capabilities_calls.append(caps)
            return
        changed = False

        # Signal.
        if changed:
            updated = dbus.Dictionary({self._self_handle: self._contact_caps[self._self_handle]},
                signature='ua(a{sv}as)')
            self.ContactCapabilitiesChanged(updated)

    def _get_contact_capabilities(self, contact):
        contact_caps = []
        contact_caps.append(self.text_chat_class)
        contact_caps.append(self.audio_chat_class)

        return contact_caps

    def _update_contact_capabilities(self, handles):
        caps = {}
        for handle in handles:
            caps[handle] = self._get_contact_capabilities(handle.contact)
            self._contact_caps[handle.id] = caps[handle] # update global dict

        ret = dbus.Dictionary(caps, signature='ua(a{sv}as)')
        self.ContactCapabilitiesChanged(ret)


    ### Initialization -------------------------------------------------------

    def _populate_capabilities(self):
        """ Add the default capabilities to all contacts in our
        contacts list."""
        handles = set([self._self_handle])
        self._add_default_capabilities(handles)
        self._update_contact_capabilities(handles)

        # These caps were updated before we were online.
        for caps in self._update_capabilities_calls:
            self.UpdateCapabilities(caps)
        self._update_capabilities_calls = []
