#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2011-2012 Canonical, Ltd.
#
# Authors:
#  Tiago Salem Herrmann <tiago.herrmann@canonical.com>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
import sys
import os
import dbus

logger = logging.getLogger('UfA.AndroidCall')

class AndroidCall():
    # call state macros
    CALL_STATE_IDLE = 0
    CALL_STATE_ACTIVE = 1
    CALL_STATE_HOLDING = 2
    CALL_STATE_DIALING = 3
    CALL_STATE_ALERTING = 4
    CALL_STATE_INCOMING = 5
    CALL_STATE_WAITING = 6
    CALL_STATE_DISCONNECTED = 7
    CALL_STATE_DISCONNECTING = 8

    props = {}

    # there are many, but we use only this state
    DISCONNECT_CAUSE_NOT_CONNECTED = 0

    def __init__(self):
        self.bus = dbus.SessionBus()
        self.init_android_call_proxy()

    def unregister_from_dbus(self):
        self.bus.remove_signal_receiver(self.AndroidCallStateChanged, 'CallStateChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )
        self.bus.remove_signal_receiver(self.AndroidIncomingCall, 'IncomingCall', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )
        self.bus.remove_signal_receiver(self.call_got_signal_owner_changed, 'NameOwnerChanged', 'org.freedesktop.DBus')
        self.bus.remove_signal_receiver(self.MuteSetChanged, 'MuteSetChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )
        self.bus.remove_signal_receiver(self.PropertiesChanged, 'PropertiesChanged', 'org.freedesktop.DBus.Properties', 'com.canonical.Android' )
        self.bus.remove_signal_receiver(self.SpeakerSetChanged, 'SpeakerSetChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )

    def connect_android_call_bus(self):
        try:
            android_object = self.bus.get_object('com.canonical.Android',
                                                    '/com/canonical/android/telephony/Telephony')
            self.android_call_proxy = dbus.Interface(android_object,

                                                    'com.canonical.android.telephony.Telephony')
            self.android_call_properties = dbus.Interface(android_object,
                                                    'org.freedesktop.DBus.Properties')
            self.props = self.android_call_properties.GetAll('com.canonical.android.telephony.Telephony')

            self.bus.add_signal_receiver(self.AndroidIncomingCall, 'IncomingCall', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android')
            self.bus.add_signal_receiver(self.AndroidCallStateChanged, 'CallStateChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android')
            self.bus.add_signal_receiver(self.MuteSetChanged, 'MuteSetChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )
            self.bus.add_signal_receiver(self.PropertiesChanged, 'PropertiesChanged', 'org.freedesktop.DBus.Properties', 'com.canonical.Android' )
            self.bus.add_signal_receiver(self.SpeakerSetChanged, 'SpeakerSetChanged', 'com.canonical.android.telephony.Telephony', 'com.canonical.Android' )

        except:
            # bus not available yet
            pass

    def init_android_call_proxy(self):
        self.connect_android_call_bus()
        self.bus.add_signal_receiver(self.call_got_signal_owner_changed, 'NameOwnerChanged', 'org.freedesktop.DBus', arg0='com.canonical.Android')

    def call_got_signal_owner_changed(self, name, old_owner, new_owner):
        if name == 'com.canonical.Android':
            if len(new_owner) != 0:
                self.connect_android_call_bus()
                self.AndroidConnected()

    def AndroidCallStateChanged(self, calls):
        pass

    def AndroidIncomingCall(self, call):
        pass

    def AndroidConnected(self):
        pass

    def acceptCall(self):
        self.android_call_proxy.acceptCall()

    def canConference(self):
        self.android_call_proxy.canConference()

    def comparePhoneNumbers(self, number, number2):
        return self.android_call_proxy.comparePhoneNumbers(number, number2)

    def createCall(self, number):
        self.android_call_proxy.createCall(number)

    def getDefaultPhone(self):
        return self.android_call_proxy.getDefaultPhone()

    def getMute(self):
        return self.android_call_proxy.getMute()

    def getVoiceMessageNumber(self):
        if not 'VoiceMailNumber' in self.props:
            return None
        return self.props['VoiceMailNumber']

    def getVoiceMessageCount(self):
        if not 'VMessageCount' in self.props:
            return 0
        return self.props['VMessageCount']

    def getVoiceMessageIndicator(self):
        if not 'MsgWaitingInd' in self.props:
            return False
        return self.props['MsgWaitingInd']

    def hangupBackgroundCall(self):
        return self.android_call_proxy.hangupBackgroundCall()

    def hangupConnection(self, connId):
        return self.android_call_proxy.hangupConnection(connId)

    def hangupForegroundCall(self):
        return self.android_call_proxy.hangupForegroundCall()

    def isSpeakerOn(self):
        return self.android_call_proxy.isSpeakerOn()

    def mergeIntoConference(self):
        return self.android_call_proxy.mergeIntoConference()
 
    def MuteSetChanged(self):
        pass

    def rejectCall(self):
        self.android_call_proxy.rejectCall()

    def restoreSpeakerMode(self):
        self.android_call_proxy.restoreSpeakerMode()

    def sendBurstDtmf(self, tone, on, off):
        self.android_call_proxy.sendBurstDtmf(tone, on, off)

    def sendDtmf(self, tone):
        self.android_call_proxy.sendDtmf(tone)

    def separateCallFromConference(self, connId):
        self.android_call_proxy.separateCallFromConference(connId)

    def setMute(self, mute):
        self.android_call_proxy.setMute(mute)

    def startDtmf(self, tone):
        self.android_call_proxy.startDtmf(tone)

    def stopDtmf(self):
        self.android_call_proxy.stopDtmf()

    def switchHoldingAndActive(self):
        self.android_call_proxy.switchHoldingAndActive()

    def turnOnSpeaker(self, flag, store):
        self.android_call_proxy.turnOnSpeaker(flag, store)

    def VoicemailIndicatorChanged(self, active):
        pass

    def VoicemailCountChanged(self, count):
        pass

    def PropertiesChanged(self, iface, changedProps, ignoredProps):
        if iface != 'com.canonical.android.telephony.Telephony':
            return

        for prop in changedProps:
            self.props[prop] = changedProps[prop]

        for prop in ignoredProps:
            del self.props[prop]

        if 'MsgWaitingInd' in changedProps:
            self.VoicemailIndicatorChanged(self.props['MsgWaitingInd'])
        if 'VMessageCount' in changedProps:
            self.VoicemailCountChanged(self.props['VMessageCount'])

    def SpeakerSetChanged(self):
        pass
