#!/usr/bin/python

# To Do:
# 1) daemonize

"""nano udev

Finds the mouse and keyboard input devices on a system where udev is not
present and creates symlinks so that X can use them easily.
"""

import os
import platform
import logging
import argparse

__author__ = "Matthew Fischer"
__version__ = "0.4"
__copyright__ = "Copyright (c) 2012 Canonical Ltd."
__license__ = "GPL"

WATCH_DIR = '/sys/class/input'
KEYBOARD_DEV = '/dev/input/keyboard'
MOUSE_DEV = '/dev/input/mouse'
X_SERVER_MT_CONFIG = '/usr/share/X11/xorg.conf.d/05-mtdev.conf'
logging.basicConfig(level=logging.WARNING)
testMode = False

def word_size():
    bits, binary_format = platform.architecture()
    if bits.lower() == '32bit':
        logging.debug("word size is 32 bits")
        return 32
    elif bits.lower() == '64bit':
        logging.debug("word size is 64 bits")
        return 64
    else:
        raise Exception('word size is an unexpected value: %s' % bits)

WORD_SIZE = word_size()

def create_mask(length):
    ret = []
    for i in range (0,length):
        ret.append(False)
    return ret

def set_mask(intval, startidx, wordsize, mask):
    for i in range (startidx,startidx+wordsize):
        if (1 << (i-startidx)) & intval:
            mask[i] = True
        else:
            mask[i] = False

# we only process in words, so masks must be a multiple
def get_mask_length(max_bit):
    if max_bit <= 0:
        return WORD_SIZE
    else:
        return ((max_bit/WORD_SIZE) + 1) * WORD_SIZE

# given > 1 external keyboard, finds the presumptive one, this is a
# hueristic only.
def find_keyboard_candidate(devices):
    candidates = []

    # whether it has LEDs is our best choice, so make a list of those
    # (we probably only have 1 with LEDS though)
    for device in devices:
        if device.has_leds():
            candidates.append(device)
            logging.debug("%s has LEDs" % device.name())

    # if none had LEDs, everything is a candidate
    if len(candidates) == 0:
        logging.warning("no LEDs found, using the device with the largest event"\
            "number")
        candidates = devices

    # use the one with the largest event number
    candidates.sort(key=lambda x: os.path.basename(x.dev_path()))
    return candidates[-1]

# given > 1 multitouch, finds the presumptive one, this is a
# hueristic only.
def find_multitouch_candidate(devices):
    candidates = devices

    # use the one with most feature set
    candidates.sort(key=lambda x: x.num_of_multitouch_features())
    return candidates[-1]

def remove_link(path):
    if os.path.islink(path):
        os.remove(path)

def make_link(source, linkname):
    try:
        if os.path.exists(linkname):
            remove_link(linkname)
        os.symlink(source, linkname)
        return True
    except Exception, e:
        logging.error("Error when making link from %s to %s: %s"
            % (source, linkname, e.message))
        return False

# These are all from /usr/include/linux/input.h
class Bits:
    # ev
    EV_KEY =        0x01
    EV_REL =        0x02
    EV_ABS =        0x03
    EV_MAX =        0x1f

    # rel
    REL_X =         0x00
    REL_Y =         0x01
    REL_MAX =       0x0f

    # abs
    ABS_X =         0x00
    ABS_Y =         0x01
    ABS_MAX =       0x3f
    ABS_MT_BEGIN1      = 0x2f
    ABS_MT_END1        = 0x3a
    ABS_MT_DISTANCE    = 0x3d

    # key
    KEY_1              = 2
    KEY_2              = 3
    KEY_3              = 4
    KEY_Q              = 16
    KEY_W              = 17
    KEY_E              = 18
    BTN_MISC           = 0x100
    BTN_MOUSE          = 0x110
    BTN_TOOL_FINGER    = 0x145
    KEY_OK             = 0x160
    BTN_TRIGGER_HAPPY  = 0x2c0
    KEY_MAX            = 0x2ff

    # led
    LED_MAX            = 0x0f

class Device:
    _path = ""
    _evmask = []
    _absmask = []
    _relmask = []
    _keymask = []
    _ledmask = []
    _n_multitouch_features = -1

    def __init__(self, path):
        self._path = path
        self._evmask = create_mask(get_mask_length(Bits.EV_MAX))
        self._absmask = create_mask(get_mask_length(Bits.ABS_MAX))
        self._relmask = create_mask(get_mask_length(Bits.REL_MAX))
        self._keymask = create_mask(get_mask_length(Bits.KEY_MAX))
        self._ledmask = create_mask(get_mask_length(Bits.LED_MAX))

    def path(self):
        return self._path

    def name(self):
        name_path = os.path.join(self._path,"device/name")
        name = ""
        if os.path.exists(name_path):
            try:
                f = open(name_path, 'r')
                name = f.read()
                f.close()
            except Exception, e:
                logging.error("Unable to get name for %s, %s" % self._path,
                    e.message)
            return name.rstrip()

    def dev_path(self):
        return os.path.join("/dev/input",os.path.basename(self._path))

    # algorithm from test_key() in udev-builtin-input_id.c in udev
    def is_keyboard(self):
        # do we have any KEY_ capability?
        if self._evmask[Bits.EV_KEY] is False:
            return False

        # bit 0 should be false (KEY_RESERVED), bits 1-31 should all be true
        if self._keymask[0] is False:
            for i in range(1,32):
                if self._keymask[i] is False:
                    return False
            return True

        else:
            return False


    # algorithm from test_pointer() in udev-builtin-input_id.c in udev
    def is_mouse(self):
        if self._keymask[Bits.BTN_TOOL_FINGER]:
            logging.debug("found a touchpad, ignoring")
            return False
        if self._keymask[Bits.BTN_MOUSE]:
            # ABS stuff seems to be for VMWare's mouse according to comments
            # in the C code, leaving here in case
            if self._evmask[Bits.EV_ABS]:
                if self._absmask[Bits.ABS_X] and self._absmask[Bits.ABS_Y]:
                    return True
            # for a real actual mouse
            if self._evmask[Bits.EV_REL]:
                if self._relmask[Bits.REL_X] and self._relmask[Bits.REL_Y]:
                    return True
        return False

    # algorithm from mtdev
    def is_multitouch(self):
        if self.num_of_multitouch_features() > 0:
            return True
        return False

    def num_of_multitouch_features(self):
        if self._n_multitouch_features != -1:
            return self._n_multitouch_features
        self._n_multitouch_features = 0
        for i in range(Bits.ABS_MT_BEGIN1, Bits.ABS_MT_END1 + 1):
            if self._absmask[i]:
                self._n_multitouch_features += 1
        # if this device only have this feature, should we drop it ?
        if self._absmask[Bits.ABS_MT_DISTANCE]: 
            self._n_multitouch_features += 1
        return self._n_multitouch_features

    def _build_cap_mask(self, val, capname):
        if capname == 'ev':
            mask = self._evmask
        elif capname == 'rel':
            mask = self._relmask
        elif capname == 'abs':
            mask = self._absmask
        elif capname == 'key':
            mask = self._keymask
        elif capname == 'led':
            mask = self._ledmask
        # this shouldn't happen, here for sanity check
        else:
            mask = None

        vals = val.split()
        i = len(vals)-1
        j = 0
        # walk backwards due to endianness
        while i>=0:
            intval = int(vals[i], 16)
            set_mask(intval, j*WORD_SIZE, WORD_SIZE, mask)
            i -= 1
            j += 1

    # return true if we have any led capability
    def has_leds(self):
        for val in self._ledmask:
            if val is True:
                return True
        return False

    def read_capabilities(self):
        """This should be called after you create the Device object so that
        the capabilities are read from /sys"""
        cap_dir = os.path.join(self._path,"device/capabilities")
        if os.path.isdir(cap_dir):
            try:
                caps = os.listdir(cap_dir)
                for cap in caps:
                    if cap in ['ev', 'abs', 'rel', 'key', 'led']:
                        cap_path = os.path.join(cap_dir, cap)
                        f = open(cap_path, 'r')
                        val = f.read().rstrip()
                        f.close()
                        self._build_cap_mask(val, cap)
            except Exception, e:
                logging.error("Error when processing capabilities for %s,"\
                    "cap=%s, %s" % (self._path, cap, e.message))

def gen_xserver_mt_config(device_node, configname):
    try:
        if os.path.exists(configname):
            os.remove(configname)
        f = open(configname, "w")

        config_data = """
Section "InputDevice"
     Identifier  "Mouse0"
     Driver      "multitouch"
     Option      "Device"     """ + '"' + device_node + '"' + """
     Option      "GrabDevice" "True"
     Option "CorePointer"
EndSection

"""
        f.write(config_data)
        f.close()
        return True
    except Exception, e:
        logging.error("Error when generating X Server configuration file (%s) on multi-touch device for device node %s: %s"
            % (configname, device_node, e.message))
        return False

def scan():
    devices = []
    keyboards = []
    mice = []
    multitouches = []

    for f in os.listdir(WATCH_DIR):
        if f.startswith('event'):
           d = Device(os.path.join(WATCH_DIR,f))
           d.read_capabilities()
           devices.append(d)
        else:
            logging.debug("Skipping %s" + f)

    for device in devices:
        if device.is_mouse():
            logging.info("%s is a mouse (%s)" % (device.dev_path(),
                device.name()))
            mice.append(device)
        if device.is_keyboard():
            logging.info("%s is a keyboard (%s)" % (device.dev_path(),
                device.name()))
            keyboards.append(device)
        if device.is_multitouch():
            logging.info("%s is a multitouch (%s)" % (device.dev_path(),
                device.name()))
            multitouches.append(device)

    real_kbd = None
    if len(keyboards) > 1:
        logging.debug("more than one keyboard, applying heuristic")
        real_kbd = find_keyboard_candidate(keyboards)
        logging.info("%s is the external keyboard candidate (%s)" %
            (real_kbd.dev_path(), real_kbd.name()))
    elif len(keyboards) == 1:
        real_kbd = keyboards[0]
    else:
        logging.error("No keyboards found!")

    # XXX - no code to handle > 1 device found for mice yet, need examples
    real_mouse = None
    if len(mice) > 0:
        real_mouse = mice[0]

    real_mt = None
    if len(multitouches) > 1:
        logging.debug("more than one multi touch devices, applying heuristic")
        real_mt = find_multitouch_candidate(multitouches)
        logging.info("%s is the external multi touch device candidate (%s)" %
            (real_mt.dev_path(), real_mt.name()))
    elif len(multitouches) == 1:
        real_mt = multitouches[0]
    else:
        logging.info("No multi-touch devices found!")

    if not testMode:
        if real_kbd:
            make_link(real_kbd.dev_path(), KEYBOARD_DEV)

        if real_mouse:
            make_link(real_mouse.dev_path(), MOUSE_DEV)

        if real_mt:
            logging.info("multi touch device path: " + real_mt.dev_path())
            # un-comment below to modify the xorg.conf
            # gen_xserver_mt_config(real_mt.dev_path(), X_SERVER_MT_CONFIG)

    else:
        if real_kbd:
            print("nudev-keyboard: PASS: %s:%s" % (real_kbd.name(),
                real_kbd.dev_path()))
        else:
            print("nudev-keyboard: FAIL: %s" % "no keyboard found")
        if real_mouse:
            print("nudev-mouse: PASS: %s:%s" % (real_mouse.name(),
                real_mouse.dev_path()))
        else:
            print("nudev-mouse: FAIL: %s" % "no mouse found")

# remove old devices
def cleanup_old_devices():
    remove_link(MOUSE_DEV)
    remove_link(KEYBOARD_DEV)

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument('--test', action='store_true')
    args = parser.parse_args()
    if args.test:
        logging.disable(logging.ERROR)
        testMode = True
    else:
        cleanup_old_devices()
    scan()
