/*
 * Copyright (C) 2008 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Neil Jagdish Patel <neil.patel@canonical.com>
 *
 */

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include "nl-sidebar-scrollbutton.h"

#include <glib.h>
#include <glib/gi18n.h>
#include <math.h>

#include "nl-config.h"
#include "nl-defines.h"
#include "nl-private.h"
#include "nl-texture-frame.h"

G_DEFINE_TYPE (NlSidebarScrollbutton, nl_sidebar_scrollbutton, CTK_TYPE_BUTTON);

#define NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj),\
  NL_TYPE_SIDEBAR_SCROLLBUTTON, \
  NlSidebarScrollbuttonPrivate))

#define AUTOSCROLL_TIME  200

struct _NlSidebarScrollbuttonPrivate
{
  CtkScrollView *scroll;
  ClutterActor  *image;
  gint           type;
  guint          autoscroll_timer;
};

enum
{
  PROP_0,

  PROP_SCROLLVIEW,
  PROP_TYPE
};


/* Globals */
static ClutterActor *normal_texture = NULL;

/* Forwards */
static void on_scroll_value_changed (CtkScrollView         *view,
                                     GParamSpec            *pspec,
                                     NlSidebarScrollbutton *self);

/* GObject stuff */
static void
nl_sidebar_scrollbutton_set_property (GObject      *object,
                                      guint         prop_id,
                                      const GValue *value,
                                      GParamSpec   *pspec)
{
  NlSidebarScrollbuttonPrivate *priv = NL_SIDEBAR_SCROLLBUTTON (object)->priv;

  switch (prop_id)
    {
    case PROP_SCROLLVIEW:
      priv->scroll = g_value_get_pointer (value);
      break;

    case PROP_TYPE:
      priv->type = g_value_get_int (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_sidebar_scrollbutton_get_property (GObject      *object,
                                      guint         prop_id,
                                      GValue       *value,
                                      GParamSpec   *pspec)
{
  NlSidebarScrollbuttonPrivate *priv;

  g_return_if_fail (NL_IS_SIDEBAR_SCROLLBUTTON (object));
  priv = NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE (object);

  switch (prop_id)
    {
    case PROP_SCROLLVIEW:
      g_value_set_pointer (value, priv->scroll);
      break;

    case PROP_TYPE:
      g_value_set_int (value, priv->type);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
    }
}

static void
nl_sidebar_scrollbutton_finalize (GObject *object)
{
  NlSidebarScrollbuttonPrivate *priv;

  priv = NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE (object);

  G_OBJECT_CLASS (nl_sidebar_scrollbutton_parent_class)->finalize (object);
}

static void
nl_sidebar_scrollbutton_get_preferred_height (ClutterActor *actor,
    gfloat        for_width,
    gfloat       *min_height,
    gfloat       *nat_height)
{
  if (min_height)
    *min_height = 24.0;
  if (nat_height)
    *nat_height = 24.0;
}

static gboolean
check_scroll_values (NlSidebarScrollbutton *self)
{
  on_scroll_value_changed (self->priv->scroll, NULL, self);

  return FALSE;
}

static void
nl_sidebar_scrollbutton_constructed (GObject *object)
{
  NlSidebarScrollbuttonPrivate *priv = NL_SIDEBAR_SCROLLBUTTON (object)->priv;
  ClutterActor *bg, *image;
  CtkPadding    padding = {  0, 24.0, 0, 0 };
  GdkPixbuf    *pixbuf;

  if (G_OBJECT_CLASS (nl_sidebar_scrollbutton_parent_class)->constructed)
    G_OBJECT_CLASS (nl_sidebar_scrollbutton_parent_class)->constructed (object);

  ctk_actor_set_padding (CTK_ACTOR (object), &padding);

  if (!CLUTTER_IS_ACTOR (normal_texture))
    {
      normal_texture = clutter_texture_new_from_file
                       (PKGDATADIR"/sidebar_normal.png", NULL);
    }

  bg = nl_texture_frame_new (CLUTTER_TEXTURE (normal_texture),
                             0, 30, 0, 0);
  ctk_actor_set_background (CTK_ACTOR (object), bg);

  pixbuf = gdk_pixbuf_new_from_file (priv->type == NL_ARROW_UP ?
                                      PKGDATADIR"/scroll_arrow_up.png"
                                      : PKGDATADIR"/scroll_arrow_down.png",
                                     NULL);
  image = priv->image = ctk_image_new_from_pixbuf (84, pixbuf);
  clutter_container_add_actor (CLUTTER_CONTAINER (object), image);
  clutter_actor_show (image);
  g_object_unref (pixbuf);

  g_signal_connect (priv->scroll, "notify::value",
                    G_CALLBACK (on_scroll_value_changed), object);
  on_scroll_value_changed (priv->scroll, NULL,
                           NL_SIDEBAR_SCROLLBUTTON (object));

  g_idle_add ((GSourceFunc)check_scroll_values, object);
}

static void
nl_sidebar_scrollbutton_clicked (CtkButton *self)
{
  NlSidebarScrollbuttonPrivate *priv = NL_SIDEBAR_SCROLLBUTTON (self)->priv;
  gfloat scroll_height;
  gfloat value;

  scroll_height = clutter_actor_get_height (CLUTTER_ACTOR (priv->scroll));

  value = ctk_scroll_view_get_value (priv->scroll);

  value += priv->type == NL_ARROW_UP ? -108/scroll_height : 108/scroll_height;

  ctk_scroll_view_set_value (priv->scroll, value);
}

static gboolean
autoscroll_timeout (ClutterActor *actor)
{
  nl_sidebar_scrollbutton_clicked (CTK_BUTTON (actor));

  return TRUE;
}

static gboolean
nl_sidebar_scrollbutton_on_enter (ClutterActor         *actor,
                                  ClutterCrossingEvent *event)
{
  NlSidebarScrollbuttonPrivate *priv;

  g_return_val_if_fail (CTK_IS_BUTTON (actor), FALSE);

  priv = NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE (actor);

  CLUTTER_ACTOR_CLASS (nl_sidebar_scrollbutton_parent_class)->enter_event (actor, event);

  priv->autoscroll_timer = g_timeout_add (AUTOSCROLL_TIME,
                                          (GSourceFunc)autoscroll_timeout,
                                          actor);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_PRELIGHT);

  return FALSE;
}

static gboolean
nl_sidebar_scrollbutton_on_leave (ClutterActor         *actor,
                                  ClutterCrossingEvent *event)
{
  NlSidebarScrollbuttonPrivate *priv;

  g_return_val_if_fail (CTK_IS_BUTTON (actor), FALSE);

  priv = NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE (actor);

  if (priv->autoscroll_timer)
    g_source_remove (priv->autoscroll_timer);

  CLUTTER_ACTOR_CLASS (nl_sidebar_scrollbutton_parent_class)->leave_event (actor, event);

  ctk_actor_set_state (CTK_ACTOR (actor), CTK_STATE_NORMAL);

  return FALSE;
}

static void
nl_sidebar_scrollbutton_class_init (NlSidebarScrollbuttonClass *klass)
{
  GObjectClass      *obj_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *act_class = CLUTTER_ACTOR_CLASS (klass);
  CtkButtonClass    *but_class = CTK_BUTTON_CLASS (klass);
  GParamSpec        *pspec;

  obj_class->finalize     = nl_sidebar_scrollbutton_finalize;
  obj_class->set_property = nl_sidebar_scrollbutton_set_property;
  obj_class->get_property = nl_sidebar_scrollbutton_get_property;
  obj_class->constructed  = nl_sidebar_scrollbutton_constructed;

  act_class->get_preferred_height =nl_sidebar_scrollbutton_get_preferred_height;
  act_class->enter_event          = nl_sidebar_scrollbutton_on_enter;
  act_class->leave_event          = nl_sidebar_scrollbutton_on_leave;

  but_class->clicked      = nl_sidebar_scrollbutton_clicked;

  /* Install properties */
  pspec = g_param_spec_pointer ("scrollview", "scrollview", "scrollview",
                                G_PARAM_CONSTRUCT | NL_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PROP_SCROLLVIEW, pspec);

  pspec = g_param_spec_int ("type", "type", "type",
                            NL_ARROW_UP, NL_ARROW_DOWN, NL_ARROW_UP,
                            G_PARAM_CONSTRUCT | NL_PARAM_READWRITE);
  g_object_class_install_property (obj_class, PROP_TYPE, pspec);

  g_type_class_add_private (obj_class, sizeof (NlSidebarScrollbuttonPrivate));
}

static void
nl_sidebar_scrollbutton_init (NlSidebarScrollbutton *self)
{
  NlSidebarScrollbuttonPrivate *priv;

  priv = self->priv = NL_SIDEBAR_SCROLLBUTTON_GET_PRIVATE (self);

  priv->autoscroll_timer = 0;

  clutter_actor_set_opacity (CLUTTER_ACTOR (self), 200);
}


/*
 * Private methods
 */
static void
on_scroll_value_changed (CtkScrollView         *view,
                         GParamSpec            *pspec,
                         NlSidebarScrollbutton *self)
{
  NlSidebarScrollbuttonPrivate *priv;
  gfloat            value;
  gfloat            hide_value;
  ClutterAnimation *anim;

  g_return_if_fail (NL_IS_SIDEBAR_SCROLLBUTTON (self));
  priv = self->priv;

  if (ctk_scroll_view_can_scroll (view))
    {
      value = ctk_scroll_view_get_value (view);
      hide_value = priv->type == NL_ARROW_UP ? 0.0 : 1.0;
      anim = clutter_actor_get_animation (priv->image);
      if (anim)
        clutter_animation_completed (anim);

      clutter_actor_animate (priv->image, CLUTTER_EASE_OUT_SINE, 100,
                             "opacity", value == hide_value ? 0 : 255,
                             NULL);
    }
  else
    {
      clutter_actor_animate (priv->image, CLUTTER_EASE_OUT_SINE, 100,
                             "opacity", 0,
                             NULL);
    }
}

/*
 * Public methods
 */
ClutterActor *
nl_sidebar_scrollbutton_new (NlArrowType    type,
                             CtkScrollView *view)
{
  g_return_val_if_fail (CTK_IS_SCROLL_VIEW (view), NULL);

  return g_object_new (NL_TYPE_SIDEBAR_SCROLLBUTTON,
                       "type", type,
                       "scrollview", view,
                       NULL);
}
