/* mozhelper: A GObject wrapper for the Mozilla Mozhelper API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <signal.h>
#include <glib.h>
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <nsXPCOM.h>
#include <nsCOMPtr.h>
#include <nsServiceManagerUtils.h>
#include <nsComponentManagerUtils.h>
#include <nsIDirectoryService.h>
#include <nsIObserverService.h>
#include <nsAppDirectoryServiceDefs.h>
#include <nsDirectoryServiceDefs.h>
#include <nsStringAPI.h>
#include <nsILocalFile.h>
#include <nsXULAppAPI.h>

#include <nsINIParser.h>

#ifdef SUPPORT_GNOME_PROXY
#include <gconf/gconf-client.h>

#define GNOME_PROXY_PATH "/system/proxy"
#define GNOME_HTTP_PROXY_PATH "/system/http_proxy"

#define MOZ_PROXY_TYPE "network.proxy.type"
#define MOZ_PROXY_TYPE_NONE 0
#define MOZ_PROXY_TYPE_MANUAL 1
#define MOZ_PROXY_TYPE_AUTO 2

#define MOZ_HTTP_PROXY "network.proxy.http"
#define MOZ_HTTP_PROXY_PORT "network.proxy.http_port"
#define MOZ_HTTP_NO_PROXY "network.proxy.no_proxies_on"

#define MOZ_FTP_PROXY "network.proxy.ftp"
#define MOZ_FTP_PROXY_PORT "network.proxy.ftp_port"

#define MOZ_SSL_PROXY "network.proxy.ssl"
#define MOZ_SSL_PROXY_PORT "network.proxy.ssl_port"

#define MOZ_SOCKS_PROXY "network.proxy.socks"
#define MOZ_SOCKS_PROXY_PORT "network.proxy.socks_port"

#define MOZ_AUTO_PROXY "network.proxy.autoconfig_url"
#endif

#include "mozhelper-service.h"
#include "mozhelper-history.h"
#include "mozhelper-prefs.h"
#include "mozhelper-error-private.h"
#include "mozhelper-event-source.h"
#include "mozhelper-enum-types.h"

#define APP_REGISTRY_NAME           NS_LITERAL_CSTRING("registry.dat")

#define PROFILE_ROOT_DIR_NAME       NS_LITERAL_CSTRING("Profiles")
#define DEFAULTS_DIR_NAME           NS_LITERAL_CSTRING("defaults")
#define DEFAULTS_PREF_DIR_NAME      NS_LITERAL_CSTRING("pref")
#define DEFAULTS_PROFILE_DIR_NAME   NS_LITERAL_CSTRING("profile")
#define RES_DIR_NAME                NS_LITERAL_CSTRING("res")
#define CHROME_DIR_NAME             NS_LITERAL_CSTRING("chrome")
#define PLUGINS_DIR_NAME            NS_LITERAL_CSTRING("plugins")
#define SEARCH_DIR_NAME             NS_LITERAL_CSTRING("searchplugins")
#define COMPONENTS_DIR_NAME         NS_LITERAL_CSTRING("components")

#define MWB_PROFILES_BASE       ".mozilla/moblin-web-browser"
#define MWB_PROFILES_INI        MWB_PROFILES_BASE "/profiles.ini"
#define MWB_DEFAULT_PROFILE     "olos99d6.default"
#define MWB_DEFAULT_PROFLIST    "[General]\n"\
                                "StartWithLastProfile=1\n"\
                                "\n"\
                                "[Profile0]\n"\
                                "Name=default\n"\
                                "IsRelative=1\n"\
                                "Path=" MWB_DEFAULT_PROFILE "\n"\
                                "Default=1\n"

class MozhelperDirectoryServiceProvider : public nsIDirectoryServiceProvider
{
public:
  MozhelperDirectoryServiceProvider ();
  virtual ~MozhelperDirectoryServiceProvider ();

  NS_DECL_ISUPPORTS
  NS_DECL_NSIDIRECTORYSERVICEPROVIDER

private:
  void InitialiseProfileDir ();

  gchar *mProfileDir;
};

NS_IMPL_ISUPPORTS1(MozhelperDirectoryServiceProvider, nsIDirectoryServiceProvider)

MozhelperDirectoryServiceProvider::MozhelperDirectoryServiceProvider ()
: mProfileDir (NULL)
{
}

MozhelperDirectoryServiceProvider::~MozhelperDirectoryServiceProvider()
{
  if (mProfileDir)
    g_free (mProfileDir);
}

void
MozhelperDirectoryServiceProvider::InitialiseProfileDir ()
{
  if (mProfileDir == NULL)
    {
      nsINIParser parser;
      PRBool buildList = PR_TRUE;
      char listProfFile[256];
      sprintf(listProfFile, "%s/" MWB_PROFILES_INI, g_get_home_dir ());

      nsresult rv = parser.Init (listProfFile);
      if (NS_SUCCEEDED(rv)) 
        {
          nsCAutoString buffer;
          nsCAutoString filePath;
          PRBool isRelative = PR_FALSE;
          unsigned int i;

          for (i = 0; /* empty */ ; i++) 
            {
              nsCAutoString profileID("Profile");
              profileID.AppendInt(i);

              rv = parser.GetString(profileID.get(), "IsRelative", buffer);
              if (NS_FAILED(rv)) break;;

              isRelative = buffer.EqualsLiteral("1");

              rv = parser.GetString(profileID.get(), "Path", filePath);
              if (NS_FAILED(rv)) {
                  NS_ERROR("Malformed profiles.ini: Path= not found");
                  continue;
              }

              rv = parser.GetString(profileID.get(), "Default", buffer);
              if (NS_SUCCEEDED(rv) && buffer.EqualsLiteral("1")) break;
            }

          const char* path = filePath.get();
          if (path) 
            {
              if (isRelative)
                  mProfileDir = g_build_filename(g_get_home_dir (), 
                                                 MWB_PROFILES_BASE, 
                                                 path, 
                                                 NULL);
              else
                  mProfileDir = g_build_filename(path, NULL);

              buildList = PR_FALSE;
            }
        } 

      if (buildList)
        {
          gchar *prifleBase = g_build_filename(g_get_home_dir (),
                                               MWB_PROFILES_BASE,
                                               NULL);
          if (!g_file_test (prifleBase, G_FILE_TEST_IS_DIR))
              g_mkdir_with_parents (prifleBase, 0755);

          FILE* writeFile = fopen(listProfFile, "w");
          if (fwrite(MWB_DEFAULT_PROFLIST, 
                     sizeof(char), 
                     sizeof (MWB_DEFAULT_PROFLIST), 
                     writeFile) != sizeof (MWB_DEFAULT_PROFLIST)) 
            {
              fclose(writeFile);
              NS_ERROR("flush profile list file failed!");
              return;
            }
          fclose(writeFile);
          mProfileDir = g_build_filename (g_get_home_dir (), 
                                          MWB_PROFILES_BASE, 
                                          MWB_DEFAULT_PROFILE, 
                                          NULL);
          if (prifleBase)
              g_free(prifleBase);
        }

      if (!g_file_test (mProfileDir, G_FILE_TEST_IS_DIR))
        g_mkdir_with_parents (mProfileDir, 0755);
    }
}

NS_IMETHODIMP
MozhelperDirectoryServiceProvider::GetFile (const char *prop,
                                         PRBool *persistent,
                                         nsIFile **retval)
{
  InitialiseProfileDir ();

  nsCOMPtr<nsILocalFile> localFile;
  nsresult rv = NS_ERROR_FAILURE;

  *retval = nsnull;
  *persistent = PR_TRUE;

  if ((strcmp (prop, NS_APP_APPLICATION_REGISTRY_DIR) == 0) ||
      (strcmp (prop, NS_APP_USER_PROFILE_50_DIR) == 0))
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (mProfileDir),
                                  1, getter_AddRefs (localFile));
    }
  else if (strcmp (prop, NS_APP_APPLICATION_REGISTRY_FILE) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (mProfileDir),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendNative (APP_REGISTRY_NAME);
    }
  else if (strcmp (prop, NS_APP_DEFAULTS_50_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (DEFAULTS_DIR_NAME);
    }
  else if (strcmp (prop, NS_APP_PREF_DEFAULTS_50_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        {
          rv = localFile->AppendRelativeNativePath (DEFAULTS_DIR_NAME);
          if (NS_SUCCEEDED (rv))
            rv = localFile->AppendRelativeNativePath (DEFAULTS_PREF_DIR_NAME);
        }
    }
  else if ((strcmp (prop, NS_APP_PROFILE_DEFAULTS_NLOC_50_DIR) == 0) ||
           (strcmp (prop, NS_APP_PROFILE_DEFAULTS_50_DIR) == 0))
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        {
          rv = localFile->AppendRelativeNativePath (DEFAULTS_DIR_NAME);
          if (NS_SUCCEEDED (rv))
            rv = localFile->
              AppendRelativeNativePath (DEFAULTS_PROFILE_DIR_NAME);
        }
    }
  else if (strcmp (prop, NS_APP_USER_PROFILES_ROOT_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (mProfileDir),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        {
          rv = localFile->AppendRelativeNativePath (PROFILE_ROOT_DIR_NAME);
          if (NS_SUCCEEDED (rv))
            {
              PRBool exists;
              rv = localFile->
                AppendRelativeNativePath (
                    nsDependentCString (g_get_user_name ()));
              rv = localFile->Exists (&exists);
              if (NS_SUCCEEDED (rv) && !exists)
                rv = localFile->Create (nsIFile::DIRECTORY_TYPE, 0755);
            }
        }
    }
  else if (strcmp (prop, NS_APP_RES_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (RES_DIR_NAME);
    }
  else if (strcmp (prop, NS_APP_CHROME_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (CHROME_DIR_NAME);
    }
  else if (strcmp (prop, NS_APP_PLUGINS_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (mProfileDir),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (PLUGINS_DIR_NAME);
    }
  else if (strcmp (prop, NS_APP_SEARCH_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (mProfileDir),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (SEARCH_DIR_NAME);
    }
  else if (strcmp (prop, NS_GRE_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
    }
  else if (strcmp (prop, NS_GRE_COMPONENT_DIR) == 0)
    {
      rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME),
                                  1, getter_AddRefs (localFile));
      if (NS_SUCCEEDED (rv))
        rv = localFile->AppendRelativeNativePath (COMPONENTS_DIR_NAME);
    }

  if (NS_SUCCEEDED (rv))
    rv = localFile->QueryInterface (nsIFile::GetIID (), (void **) retval);

  if (NS_SUCCEEDED (rv))
    {
      nsAutoString ns_path;
      char *path;
      localFile->GetPath (ns_path);
      path = ToNewUTF8String (ns_path);
      g_debug ("Directory provider returning '%s' for '%s'", path, prop);
      NS_Free (path);
    }
  else
    g_debug ("Failed to get directory for '%s'", prop);

  return rv;
}

#ifdef SUPPORT_GNOME_PROXY
static void
mozhelper_service_gconf_cb (GConfClient *client,
                      guint        cnxn_id,
                      GConfEntry  *entry,
                      gpointer     user_data)
{
  /* Copy Gnome proxy settings over Mozilla proxy settings */
  g_debug ("Synchronising proxy settings");

  nsresult rv;
  nsCOMPtr<nsIPrefService> prefService
    = do_GetService (NS_PREFSERVICE_CONTRACTID, &rv);
  if (NS_FAILED (rv))
    return;
  nsCOMPtr<nsIPrefBranch> rootBranch;
  rv = prefService->GetDefaultBranch ("", getter_AddRefs (rootBranch));
  if (NS_FAILED (rv))
    return;

  if (rootBranch)
    {
      gchar *mode = gconf_client_get_string (client,
                                             GNOME_PROXY_PATH "/mode",
                                             NULL);
      if (g_strcmp0 (mode, "manual") == 0)
        {
          /* Enable manual proxy settings */
          rootBranch->SetIntPref (MOZ_PROXY_TYPE, MOZ_PROXY_TYPE_MANUAL);

          /* Find out whether to duplicate values across proxy types */
          gboolean copy_to_others =
            gconf_client_get_bool (client,
                                   GNOME_HTTP_PROXY_PATH "/use_same_proxy",
                                   NULL);

          /* Set http proxy host url */
          gchar *string = gconf_client_get_string (client,
                                                   GNOME_HTTP_PROXY_PATH
                                                     "/host",
                                                   NULL);
          rootBranch->SetCharPref (MOZ_HTTP_PROXY, string ? string : "");
          if (copy_to_others)
            {
              rootBranch->SetCharPref (MOZ_FTP_PROXY, string ? string : "");
              rootBranch->SetCharPref (MOZ_SSL_PROXY, string ? string : "");
              rootBranch->SetCharPref (MOZ_SOCKS_PROXY, string ? string : "");
            }
          g_free (string);

          /* Set http proxy port */
          gint number = gconf_client_get_int (client,
                                              GNOME_HTTP_PROXY_PATH "/port",
                                              NULL);
          rootBranch->SetIntPref (MOZ_HTTP_PROXY_PORT, number);
          if (copy_to_others)
            {
              rootBranch->SetIntPref (MOZ_FTP_PROXY_PORT, number);
              rootBranch->SetIntPref (MOZ_SSL_PROXY_PORT, number);
              rootBranch->SetIntPref (MOZ_SOCKS_PROXY_PORT, number);
            }

          /* Set http proxy exclusion list */
          GSList *list = gconf_client_get_list (client,
                                                GNOME_HTTP_PROXY_PATH
                                                  "/ignore_hosts",
                                                GCONF_VALUE_STRING,
                                                NULL);
          if (list)
            {
              GSList *u;
              GString *no_proxy = g_string_new ("");

              for (u = list; u; u = u->next)
                {
                  gchar *url = (gchar *)u->data;
                  g_string_append (no_proxy, url);
                  if (u->next)
                    g_string_append (no_proxy, ", ");
                  g_free (url);
                }
              g_slist_free (list);

              rootBranch->SetCharPref (MOZ_HTTP_NO_PROXY,
                                       no_proxy->str);
              g_string_free (no_proxy, TRUE);
            }
          else
            {
              rootBranch->SetCharPref (MOZ_HTTP_NO_PROXY, "");
            }

          if (!copy_to_others)
            {
              /* Set ftp proxy host url */
              string = gconf_client_get_string (client,
                                                GNOME_PROXY_PATH "/ftp_host",
                                                NULL);
              rootBranch->SetCharPref (MOZ_FTP_PROXY, string ? string : "");
              g_free (string);

              /* Set ftp proxy port */
              number = gconf_client_get_int (client,
                                             GNOME_PROXY_PATH "/ftp_port",
                                             NULL);
              rootBranch->SetIntPref (MOZ_FTP_PROXY_PORT, number);

              /* Set ssl proxy host url */
              string = gconf_client_get_string (client,
                                                GNOME_PROXY_PATH "/secure_host",
                                                NULL);
              rootBranch->SetCharPref (MOZ_SSL_PROXY, string ? string : "");
              g_free (string);

              /* Set ssl proxy port */
              number = gconf_client_get_int (client,
                                             GNOME_PROXY_PATH "/secure_port",
                                             NULL);
              rootBranch->SetIntPref (MOZ_SSL_PROXY_PORT, number);

              /* Set socks proxy host url */
              string = gconf_client_get_string (client,
                                                GNOME_PROXY_PATH "/socks_host",
                                                NULL);
              rootBranch->SetCharPref (MOZ_SOCKS_PROXY, string ? string : "");
              g_free (string);

              /* Set socks proxy port */
              number = gconf_client_get_int (client,
                                             GNOME_PROXY_PATH "/socks_port",
                                             NULL);
              rootBranch->SetIntPref (MOZ_SOCKS_PROXY_PORT, number);
            }
        }
      else if (g_strcmp0 (mode, "auto") == 0)
        {
          /* Enable manual proxy settings */
          rootBranch->SetIntPref (MOZ_PROXY_TYPE, MOZ_PROXY_TYPE_AUTO);

          gchar *string = gconf_client_get_string (client,
                                                   GNOME_PROXY_PATH
                                                     "/autoconfig_url",
                                                   NULL);
          rootBranch->SetCharPref (MOZ_AUTO_PROXY, string ? string : "");
          g_free (string);
        }
      else
        {
          /* Disable proxy settings */
          rootBranch->SetIntPref (MOZ_PROXY_TYPE, MOZ_PROXY_TYPE_NONE);
        }

      g_free (mode);
    }
}
#endif

static GMainLoop *main_loop = NULL;

static gboolean
mozhelper_quit_cb (gpointer data)
{
  g_main_loop_quit (main_loop);
  return FALSE;
}

static void
mozhelper_signal_cb (int sig)
{
  g_idle_add_full (G_PRIORITY_HIGH, mozhelper_quit_cb, NULL, NULL);
}

static void
mozhelper_service_load_thebes (void)
{
  /* This creates a temporary Thebes image just to ensure that the
     Thebes modules has been loaded. Otherwise when we try to
     decompress a PNG image (such as from a favicon) Mozilla will
     directly try to use gfxPlatform::GetPlatform() but nothing will
     have created the platform instance and MOZHELPER will crash. See Moblin
     bug #5012 for more information */
  nsCOMPtr<nsISupports> image;
  image = do_CreateInstance ("@mozilla.org/gfx/image;1");
}

static void
mozhelper_service_run_main_loop (GError **error)
{
  nsresult rv;

  nsCOMPtr<nsILocalFile> moz_home;
  rv = NS_NewNativeLocalFile (nsDependentCString (MOZHOME), PR_TRUE,
                              getter_AddRefs (moz_home));
  if (NS_FAILED (rv))
    mozhelper_error_set_from_nsresult (rv, error);
  else
    {
      /* Initialize embedding */
      nsIDirectoryServiceProvider *dir_provider
        = new MozhelperDirectoryServiceProvider;
      /* The dir_provider has no reference so XPCom will take
         ownership of it */
      rv = XRE_InitEmbedding (moz_home, moz_home, dir_provider,
                              nsnull, nsnull);

      if (NS_FAILED (rv))
        mozhelper_error_set_from_nsresult (rv, error);
      else
        {
          MozhelperHistory *history;
          MozhelperPrefs *prefs;

          GSource *event_source = mozhelper_event_source_new ();

          XRE_NotifyProfile ();

          /* Make sure the objects we created are unrefd before
             shutting down XPCom */
          moz_home = 0;

          history = mozhelper_history_new ();
          prefs = mozhelper_prefs_new ();

#ifdef SUPPORT_GNOME_PROXY
          GConfClient *client = gconf_client_get_default ();
          gconf_client_add_dir (client,
                                GNOME_PROXY_PATH,
                                GCONF_CLIENT_PRELOAD_ONELEVEL,
                                NULL);
          gconf_client_add_dir (client,
                                GNOME_HTTP_PROXY_PATH,
                                GCONF_CLIENT_PRELOAD_ONELEVEL,
                                NULL);
          gconf_client_notify_add (client,
                                   GNOME_PROXY_PATH,
                                   mozhelper_service_gconf_cb,
                                   prefs,
                                   NULL,
                                   NULL);
          gconf_client_notify_add (client,
                                   GNOME_HTTP_PROXY_PATH,
                                   mozhelper_service_gconf_cb,
                                   prefs,
                                   NULL,
                                   NULL);
          mozhelper_service_gconf_cb (client, 0, NULL, prefs);
#endif

          g_source_attach (event_source, NULL);

          mozhelper_service_load_thebes ();

          signal (SIGINT, mozhelper_signal_cb);
          signal (SIGTERM, mozhelper_signal_cb);
          g_main_loop_run (main_loop);

          /* Let the services know the application is shutting down */
          nsCOMPtr<nsIObserverService> observer_service
            = do_GetService ("@mozilla.org/observer-service;1", &rv);
          if (NS_SUCCEEDED (rv))
            {
              NS_NAMED_LITERAL_STRING (shutdown_str, "shutdown");
              observer_service->NotifyObservers (nsnull,
                                                 "quit-application",
                                                 shutdown_str.get ());
            }
          observer_service = nsnull;

          g_object_unref (G_OBJECT (prefs));
          g_object_unref (G_OBJECT (history));

          g_source_destroy (event_source);
          g_source_unref (event_source);

          XRE_TermEmbedding ();
        }
    }
}

static gboolean
mozhelper_register_dbus_object (DBusGProxy   *proxy,
                             const gchar  *service)
{
  GError *error = NULL;
  guint32 request_status;

  if (!org_freedesktop_DBus_request_name (proxy,
                                          service,
                                          DBUS_NAME_FLAG_DO_NOT_QUEUE,
                                          &request_status, &error))
    {
      g_warning ("Failed to request name: %s", error->message);
      g_error_free (error);
    }
  else if (request_status == DBUS_REQUEST_NAME_REPLY_EXISTS)
    {
      g_warning ("'%s' service already exists", service);
    }
  else if (request_status != DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER)
    {
      g_warning ("Failed to request name");
    }
  else
    return TRUE;

  return FALSE;
}

static DBusHandlerResult
mozhelper_dbus_filter (DBusConnection *connection,
                 DBusMessage *message,
                 void *user_data)
{
  if (dbus_message_is_signal (message,
                              DBUS_INTERFACE_LOCAL,
                              "Disconnected"))
    {
      GMainLoop *main_loop_data = (GMainLoop *) user_data;
      g_main_loop_quit (main_loop_data);
      return DBUS_HANDLER_RESULT_HANDLED;
    }

  return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

int
main (int argc, char **argv)
{
  int ret = MOZHELPER_SERVICE_EXIT_SUCCESS;
  GError *error = NULL;
  DBusGConnection *bus;

  g_thread_init (NULL);
  g_type_init ();

  main_loop = g_main_loop_new (NULL, FALSE);

  if ((bus = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("%s", error->message);
      g_clear_error (&error);
      ret = MOZHELPER_SERVICE_EXIT_ERROR;
    }
  else
    {
      DBusGProxy *proxy;
      DBusConnection *dcon;

      /* Don't exit on disconnect */
      dcon = dbus_g_connection_get_connection (bus);
      dbus_connection_set_exit_on_disconnect (dcon, FALSE);
      /* Install a message filter to handle the disconnect signal */
      dbus_connection_add_filter (dcon, mozhelper_dbus_filter,
                                  g_main_loop_ref (main_loop),
                                  (DBusFreeFunction) g_main_loop_unref);

      proxy = dbus_g_proxy_new_for_name (bus,
                                         DBUS_SERVICE_DBUS,
                                         DBUS_PATH_DBUS,
                                         DBUS_INTERFACE_DBUS);

      if (!mozhelper_register_dbus_object (proxy, MOZHELPER_SERVICE_HISTORY))
        ret = MOZHELPER_SERVICE_EXIT_ERROR;
      else if (!mozhelper_register_dbus_object (proxy, MOZHELPER_SERVICE_PREFS))
        ret = MOZHELPER_SERVICE_EXIT_ERROR;
      else if (!mozhelper_register_dbus_object (proxy, MOZHELPER_SERVICE_COOKIES))
        ret = MOZHELPER_SERVICE_EXIT_ERROR;
      else if (!mozhelper_register_dbus_object (proxy,
                                          MOZHELPER_SERVICE_LOGIN_MANAGER_STORAGE))
        ret = MOZHELPER_SERVICE_EXIT_ERROR;
      else if (!mozhelper_register_dbus_object (proxy,
                                          MOZHELPER_SERVICE_PERMISSION_MANAGER))
        ret = MOZHELPER_SERVICE_EXIT_ERROR;
      else
        {
          dbus_g_error_domain_register (MOZHELPER_ERROR, NULL, MOZHELPER_TYPE_ERROR);

          mozhelper_service_run_main_loop (&error);

          if (error)
            {
              g_warning ("%s", error->message);
              g_error_free (error);
              ret = MOZHELPER_SERVICE_EXIT_ERROR;
            }
        }

      g_object_unref (proxy);

      dbus_g_connection_unref (bus);
    }

  g_main_loop_unref (main_loop);

  return ret;
}
