/* mozhelper: A GObject wrapper for the Mozilla Mozhelper API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib-object.h>
#include <dbus/dbus-glib.h>
#include <new>
#include <nsILocalFile.h>
#include <nsIPrefService.h>
#include <nsIPrefBranch2.h>
#include <nsIIOService.h>
#include <nsIObserver.h>
#include <nsToolkitCompsCID.h>
#include <nsNetCID.h>
#include <nsServiceManagerUtils.h>
#include <nsCOMPtr.h>
#include <nsStringAPI.h>
#include <nsMemory.h>

#include "mozhelper-prefs.h"
#include "mozhelper-error-private.h"
#include "mozhelper-service.h"
#include "mozhelper-marshal.h"

/* D-Bus method implementations */

static gboolean mozhelper_prefs_read_user (MozhelperPrefs  *self,
                                     const gchar  *path,
                                     GError      **error);
static gboolean mozhelper_prefs_reset (MozhelperPrefs  *self,
                                 GError      **error);
static gboolean mozhelper_prefs_reset_user (MozhelperPrefs  *self,
                                      GError      **error);
static gboolean mozhelper_prefs_save_pref_file (MozhelperPrefs  *self,
                                          const gchar  *path,
                                          GError      **error);
static gboolean mozhelper_prefs_get_branch (MozhelperPrefs  *self,
                                      const gchar  *root,
                                      gint         *id,
                                      GError      **error);
static gboolean mozhelper_prefs_get_default_branch (MozhelperPrefs  *self,
                                              const gchar  *root,
                                              gint         *id,
                                              GError      **error);
static gboolean mozhelper_prefs_release_branch (MozhelperPrefs  *self,
                                          gint          id,
                                          GError      **error);

static gboolean mozhelper_prefs_branch_get_type (MozhelperPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gint         *type,
                                           GError      **error);

static gboolean mozhelper_prefs_branch_get_bool (MozhelperPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gboolean     *value,
                                           GError      **error);

static gboolean mozhelper_prefs_branch_set_bool (MozhelperPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gboolean      value,
                                           GError      **error);

static gboolean mozhelper_prefs_branch_get_int (MozhelperPrefs  *self,
                                          gint          id,
                                          const gchar  *name,
                                          gint         *value,
                                          GError      **error);

static gboolean mozhelper_prefs_branch_set_int (MozhelperPrefs  *self,
                                          gint          id,
                                          const gchar  *name,
                                          gint          value,
                                          GError      **error);

static gboolean mozhelper_prefs_branch_get_char (MozhelperPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           gchar       **value,
                                           GError      **error);

static gboolean mozhelper_prefs_branch_set_char (MozhelperPrefs  *self,
                                           gint          id,
                                           const gchar  *name,
                                           const gchar  *value,
                                           GError      **error);

static gboolean mozhelper_prefs_branch_has_user_value (MozhelperPrefs  *self,
                                                 gint          id,
                                                 const gchar  *name,
                                                 gboolean     *result,
                                                 GError      **error);

static gboolean mozhelper_prefs_branch_lock (MozhelperPrefs  *self,
                                       gint          id,
                                       const gchar  *name,
                                       GError      **error);

static gboolean mozhelper_prefs_branch_is_locked (MozhelperPrefs  *self,
                                            gint          id,
                                            const gchar  *name,
                                            gboolean     *result,
                                            GError      **error);

static gboolean mozhelper_prefs_branch_unlock (MozhelperPrefs  *self,
                                         gint          id,
                                         const gchar  *name,
                                         GError      **error);

static gboolean mozhelper_prefs_branch_get_child_list (MozhelperPrefs   *self,
                                                 gint           id,
                                                 const gchar   *start,
                                                 guint         *len,
                                                 gchar       ***array,
                                                 GError       **error);

static gboolean mozhelper_prefs_branch_add_observer (MozhelperPrefs  *self,
                                               gint          id,
                                               const gchar  *domain,
                                               GError      **error);

static gboolean mozhelper_prefs_branch_remove_observer (MozhelperPrefs  *self,
                                                  gint          id,
                                                  const gchar  *domain,
                                                  GError      **error);

#include "mozhelper-prefs-glue.h"

/* End D-Bus method implementations */

static void mozhelper_prefs_finalize (GObject *object);

G_DEFINE_TYPE (MozhelperPrefs, mozhelper_prefs, G_TYPE_OBJECT);

#define MOZHELPER_PREFS_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MOZHELPER_TYPE_PREFS, \
                                MozhelperPrefsPrivate))

class MozhelperPrefsObserver : public nsIObserver
{
public:
  NS_DECL_ISUPPORTS
  NS_DECL_NSIOBSERVER

  MozhelperPrefs *mPrefs;
};

struct _MozhelperPrefsPrivate
{
  nsIPrefService   *pref_service;
  GHashTable       *branches_by_id;
  GHashTable       *ids_by_branch;
  MozhelperPrefsObserver  observer;
};

enum
  {
    READ_SIGNAL,
    RESET_SIGNAL,
    APP_DEFAULTS_SIGNAL,
    BRANCH_CHANGED_SIGNAL,

    LAST_SIGNAL
  };

static guint prefs_signals[LAST_SIGNAL] = { 0, };

NS_IMPL_QUERY_INTERFACE1(MozhelperPrefsObserver, nsIObserver)

/* We don't want to implement reference counting for
   MozhelperPrefsObserver because it is entirely owned by the
   MozhelperPrefs instance */
NS_IMETHODIMP_(nsrefcnt)
MozhelperPrefsObserver::AddRef ()
{
  return 1;
}

NS_IMETHODIMP_(nsrefcnt)
MozhelperPrefsObserver::Release ()
{
  return 1;
}

NS_IMETHODIMP
MozhelperPrefsObserver::Observe(nsISupports     *aSubject,
                          const char      *aTopic,
                          const PRUnichar *someData)
{
  MozhelperPrefsPrivate *priv = mPrefs->priv;
  nsIPrefBranch *branch = static_cast<nsIPrefBranch *>(aSubject);
  gint id = GPOINTER_TO_INT (
    g_hash_table_lookup (priv->ids_by_branch, branch));
  NS_ConvertUTF16toUTF8 domain (someData);
  g_signal_emit (mPrefs, prefs_signals[BRANCH_CHANGED_SIGNAL], 0,
                 id, domain.get ());
  return NS_OK;
}

static void
mozhelper_prefs_class_init (MozhelperPrefsClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->finalize = mozhelper_prefs_finalize;

  prefs_signals[READ_SIGNAL] =
    g_signal_new ("read",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperPrefsClass, read),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[RESET_SIGNAL] =
    g_signal_new ("reset",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperPrefsClass, reset),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[APP_DEFAULTS_SIGNAL] =
    g_signal_new ("app-defaults",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperPrefsClass, app_defaults),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);

  prefs_signals[BRANCH_CHANGED_SIGNAL] =
    g_signal_new ("branch-changed",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperPrefsClass, branch_changed),
                  NULL, NULL,
                  mozhelper_marshal_VOID__INT_STRING,
                  G_TYPE_NONE, 2, G_TYPE_INT, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MozhelperPrefsPrivate));

  dbus_g_object_type_install_info (MOZHELPER_TYPE_PREFS,
                                   &dbus_glib_mozhelper_prefs_object_info);
}

static void
mozhelper_prefs_init (MozhelperPrefs *self)
{
  nsresult rv;
  MozhelperPrefsPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MOZHELPER_PREFS_GET_PRIVATE (self);

  // 'placement new' to call the constructor for MozhelperPrefsPrivate
  new (reinterpret_cast<void *> (priv)) MozhelperPrefsPrivate;
  priv->observer.mPrefs = self;

  nsCOMPtr<nsIPrefService> pref_service
    = do_GetService (NS_PREFSERVICE_CONTRACTID, &rv);
  if (NS_SUCCEEDED (rv))
    {
      priv->pref_service = pref_service;
      NS_ADDREF (pref_service);
    }
  else
    g_warning ("Failed to retrieve preferences service object");

  priv->branches_by_id = g_hash_table_new (g_direct_hash, g_direct_equal);
  priv->ids_by_branch = g_hash_table_new (g_direct_hash, g_direct_equal);

  // Initialise the root branch
  mozhelper_prefs_get_branch (self, NULL, NULL, NULL);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      dbus_g_connection_register_g_object (connection,
                                           MOZHELPER_SERVICE_PREFS_PATH,
                                           G_OBJECT (self));
      dbus_g_connection_unref (connection);
    }
}

static void
mozhelper_prefs_finalize (GObject *object)
{
  MozhelperPrefs *self = MOZHELPER_PREFS (object);
  MozhelperPrefsPrivate *priv = self->priv;
  GHashTableIter iter;
  gpointer key, value;

  // Save prefs
  mozhelper_prefs_save_pref_file (self, NULL, NULL);

  // Unref branches and get rid of hash-table
  g_hash_table_iter_init (&iter, priv->branches_by_id);
  while (g_hash_table_iter_next (&iter, &key, &value))
    {
      nsIPrefBranch *branch = (nsIPrefBranch *)value;
      NS_RELEASE (branch);
    }
  g_hash_table_destroy (priv->branches_by_id);
  g_hash_table_destroy (priv->ids_by_branch);

  // Release reference on preferences service
  if (priv->pref_service)
    NS_RELEASE (priv->pref_service);

  /* Explicitly call the destructor for the private data (so that it
     destructs without trying to free the memory) */
  priv->~MozhelperPrefsPrivate ();

  G_OBJECT_CLASS (mozhelper_prefs_parent_class)->finalize (object);
}

MozhelperPrefs *
mozhelper_prefs_new (void)
{
  MozhelperPrefs *self = (MozhelperPrefs *)g_object_new (MOZHELPER_TYPE_PREFS, NULL);
  return self;
}

nsIPrefService *
mozhelper_prefs_get_service (MozhelperPrefs *prefs)
{
  return prefs->priv->pref_service;
}

static nsCOMPtr<nsIFile>
mozhelper_prefs_file_from_path (const gchar *path, GError **error)
{
  nsresult rv;
  nsCOMPtr<nsILocalFile> local_file =
    do_GetService ("@mozilla.org/file/local;1", &rv);

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return NULL;
    }

  rv = local_file->InitWithPath (NS_ConvertUTF8toUTF16 (path));

  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return NULL;
    }

  nsCOMPtr<nsIFile> file = do_QueryInterface (local_file);

  return file;
}

static gboolean
mozhelper_prefs_read_user (MozhelperPrefs  *self,
                        const gchar  *path,
                        GError      **error)
{
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsCOMPtr<nsIFile> ns_file = mozhelper_prefs_file_from_path (path,
                                                           error);

  if (ns_file)
    {
      nsresult rv = priv->pref_service->ReadUserPrefs (ns_file);
      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }
      else
        return TRUE;
    }
  else
    return FALSE;
}

static gboolean
mozhelper_prefs_reset (MozhelperPrefs  *self,
                    GError      **error)
{
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->ResetPrefs ();
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }
  return TRUE;
}

static gboolean
mozhelper_prefs_reset_user (MozhelperPrefs  *self,
                         GError      **error)
{
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->ResetUserPrefs ();
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }
  else
    return TRUE;
}

static gboolean
mozhelper_prefs_save_pref_file (MozhelperPrefs  *self,
                          const gchar  *path,
                          GError      **error)
{
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsCOMPtr<nsIFile> ns_file = nsnull;

  if (path && (path[0] != '\0'))
    ns_file = mozhelper_prefs_file_from_path (path,
                                        error);

  if (!path || (path[0] == '\0') || ns_file)
    {
      nsresult rv = priv->pref_service->SavePrefFile (ns_file);
      if (NS_FAILED (rv))
        {
          mozhelper_error_set_from_nsresult (rv, error);
          return FALSE;
        }
      else
        return TRUE;
    }
  else
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_UNEXPECTED, error);
      return FALSE;
    }
}

static gint
mozhelper_prefs_get_new_id ()
{
  static gint id = 0;
  return id ++;
}

static gboolean
mozhelper_prefs_get_branch (MozhelperPrefs  *self,
                      const gchar  *root,
                      gint         *id,
                      GError      **error)
{
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  // Cache the root branch
  if ((!root) || (root[0] == '\0'))
    {
      branch = (nsIPrefBranch *)
        g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (0));
      if (branch)
        {
          NS_ADDREF (branch);
          if (id)
            *id = 0;
          return TRUE;
        }
    }

  nsresult rv = priv->pref_service->GetBranch (root, &branch);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  gint new_id = mozhelper_prefs_get_new_id ();
  g_hash_table_insert (priv->branches_by_id, GINT_TO_POINTER (new_id), branch);
  g_hash_table_insert (priv->ids_by_branch, branch, GINT_TO_POINTER (new_id));

  if (id)
    *id = new_id;

  return TRUE;
}

static gboolean
mozhelper_prefs_get_default_branch (MozhelperPrefs  *self,
                                 const gchar  *root,
                                 gint         *id,
                                 GError      **error)
{
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  if (!priv->pref_service)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  nsresult rv = priv->pref_service->GetDefaultBranch (root, &branch);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  gint new_id = mozhelper_prefs_get_new_id ();
  g_hash_table_insert (priv->branches_by_id, GINT_TO_POINTER (new_id), branch);

  if (id)
    *id = new_id;

  return TRUE;
}

static gboolean
mozhelper_prefs_release_branch (MozhelperPrefs  *self,
                             gint          id,
                             GError      **error)
{
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  // We don't remove the root branch
  if (id == 0)
    return TRUE;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  g_hash_table_remove (priv->branches_by_id, GINT_TO_POINTER (id));
  g_hash_table_remove (priv->ids_by_branch, branch);
  NS_RELEASE (branch);

  return TRUE;
}

// nsPrefBranch accessor methods

static gboolean
mozhelper_prefs_branch_get_type (MozhelperPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gint         *type,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 value;
  rv = branch->GetPrefType (name, &value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (type)
    *type = value;

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_get_bool (MozhelperPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean     *value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 pr_value;
  rv = branch->GetBoolPref (name, &pr_value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = pr_value;

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_set_bool (MozhelperPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              gboolean      value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetBoolPref (name, value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_get_int (MozhelperPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint         *value,
                             GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRInt32 pr_value;
  rv = branch->GetIntPref (name, &pr_value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = pr_value;

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_set_int (MozhelperPrefs  *self,
                             gint          id,
                             const gchar  *name,
                             gint          value,
                             GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetIntPref (name, value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_get_char (MozhelperPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              char        **value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  char *ns_value;
  rv = branch->GetCharPref (name, &ns_value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (value)
    *value = g_strdup (ns_value);

  NS_Free (ns_value);

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_set_char (MozhelperPrefs  *self,
                              gint          id,
                              const gchar  *name,
                              const gchar  *value,
                              GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->SetCharPref (name, value);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_has_user_value (MozhelperPrefs  *self,
                                    gint          id,
                                    const gchar  *name,
                                    gboolean     *result,
                                    GError      **error)
{
  nsresult rv;
  PRBool retval;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->PrefHasUserValue (name, &retval);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (result)
    *result = retval;

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_lock (MozhelperPrefs  *self,
                          gint          id,
                          const gchar  *name,
                          GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->LockPref (name);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_is_locked (MozhelperPrefs  *self,
                               gint          id,
                               const gchar  *name,
                               gboolean     *result,
                               GError      **error)
{
  nsresult rv;
  PRBool retval;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->PrefIsLocked (name, &retval);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (result)
    *result = retval;

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_unlock (MozhelperPrefs  *self,
                            gint          id,
                            const gchar  *name,
                            GError      **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->UnlockPref (name);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_get_child_list (MozhelperPrefs   *self,
                                    gint           id,
                                    const gchar   *start,
                                    guint         *len,
                                    gchar       ***array,
                                    GError       **error)
{
  nsresult rv;
  nsIPrefBranch *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  PRUint32 count;
  char **ns_array;
  rv = branch->GetChildList (start, &count, &ns_array);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  if (len)
    *len = count;
  if (array)
    {
      *array = (gchar **)g_malloc0 ((count + 1) * sizeof (gchar *));
      for (PRUint32 i = 0; i < count; i++)
        (*array)[i] = g_strdup (ns_array[i]);
    }

  NS_FREE_XPCOM_ALLOCATED_POINTER_ARRAY (count, ns_array);

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_add_observer (MozhelperPrefs     *self,
                               gint          id,
                               const gchar  *domain,
                               GError      **error)
{
  nsresult rv;
  nsIPrefBranch2 *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch2 *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->AddObserver (domain,
                            static_cast<nsIObserver *>(&priv->observer),
                            PR_FALSE);
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

static gboolean
mozhelper_prefs_branch_remove_observer (MozhelperPrefs     *self,
                                  gint          id,
                                  const gchar  *domain,
                                  GError      **error)
{
  nsresult rv;
  nsIPrefBranch2 *branch;
  MozhelperPrefsPrivate *priv = self->priv;

  branch = (nsIPrefBranch2 *)
    g_hash_table_lookup (priv->branches_by_id, GINT_TO_POINTER (id));

  if (!branch)
    {
      mozhelper_error_set_from_nsresult (NS_ERROR_NOT_AVAILABLE, error);
      return FALSE;
    }

  rv = branch->RemoveObserver (domain,
                               static_cast<nsIObserver *>(&priv->observer));
  if (NS_FAILED (rv))
    {
      mozhelper_error_set_from_nsresult (rv, error);
      return FALSE;
    }

  return TRUE;
}

