/* mozhelper: A GObject wrapper for the Mozilla Mozhelper API
 *
 * Copyright (C) 2009  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib.h>

#include "mozhelper-error.h"
#include "mozhelper-error-private.h"

#include "mozhelper-history.h"
#include "mozhelper-history-bindings.h"
#include "mozhelper-service.h"
#include "mozhelper-marshal.h"

static void mozhelper_history_dispose (GObject *object);

G_DEFINE_TYPE (MozhelperHistory, mozhelper_history, G_TYPE_OBJECT);

#define MOZHELPER_HISTORY_GET_PRIVATE(obj) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((obj), MOZHELPER_TYPE_HISTORY, \
                                MozhelperHistoryPrivate))

typedef struct _MozhelperHistoryClosure MozhelperHistoryClosure;

struct _MozhelperHistoryPrivate
{
  DBusGProxy *proxy;

  guint callback_id;

  GSList *callbacks;
};

struct _MozhelperHistoryClosure
{
  MozhelperHistory *history;
  DBusGProxyCall *proxy_call;
  guint id;

  gpointer callback;
  gpointer user_data;
  GDestroyNotify user_data_notify;
};

enum
  {
    AC_RESULT_RECEIVED_SIGNAL,
    LINK_VISITED_SIGNAL,
    FAVORITES_RECEIVED_SIGNAL,
    PINNED_PAGE_SIGNAL,
    UNPINNED_PAGE_SIGNAL,

    LAST_SIGNAL
  };

static guint history_signals[LAST_SIGNAL] = { 0, };

static void
mozhelper_history_class_init (MozhelperHistoryClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  gobject_class->dispose = mozhelper_history_dispose;

  history_signals[AC_RESULT_RECEIVED_SIGNAL] =
    g_signal_new ("ac-result-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, ac_result_received),
                  NULL, NULL,
                  mozhelper_marshal_VOID__UINT_STRING_STRING,
                  G_TYPE_NONE, 3,
                  G_TYPE_UINT,
                  G_TYPE_STRING,
                  G_TYPE_STRING);

  history_signals[LINK_VISITED_SIGNAL] =
    g_signal_new ("link-visited",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, link_visited),
                  NULL, NULL,
                  mozhelper_marshal_VOID__STRING_INT,
                  G_TYPE_NONE, 2, G_TYPE_STRING, G_TYPE_INT);
  history_signals[FAVORITES_RECEIVED_SIGNAL] =
    g_signal_new ("favorites-received",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, favorites_received),
                  NULL, NULL,
                  mozhelper_marshal_VOID__BOXED_BOXED,
                  G_TYPE_NONE, 2, G_TYPE_STRV, G_TYPE_STRV);

  history_signals[PINNED_PAGE_SIGNAL] =
    g_signal_new ("pinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, pinned_page),
                  NULL, NULL,
                  mozhelper_marshal_VOID__STRING_STRING_INT_BOOL,
                  G_TYPE_NONE, 4,
                  G_TYPE_STRING, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);

  history_signals[UNPINNED_PAGE_SIGNAL] =
    g_signal_new ("unpinned-page",
                  G_TYPE_FROM_CLASS (gobject_class),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MozhelperHistoryClass, unpinned_page),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (MozhelperHistoryPrivate));

  dbus_g_object_register_marshaller (mozhelper_marshal_VOID__UINT_STRING_STRING,
                                     G_TYPE_NONE,
                                     G_TYPE_UINT,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mozhelper_marshal_VOID__BOXED_BOXED,
                                     G_TYPE_NONE,
                                     G_TYPE_STRV,
                                     G_TYPE_STRV,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mozhelper_marshal_VOID__STRING_INT,
                                     G_TYPE_NONE,
                                     G_TYPE_STRING,
                                     G_TYPE_INT,
                                     G_TYPE_INVALID);
  dbus_g_object_register_marshaller (mozhelper_marshal_VOID__STRING_STRING_INT_BOOL,
                                     G_TYPE_NONE,
                                     G_TYPE_STRING,
                                     G_TYPE_STRING,
                                     G_TYPE_INT,
                                     G_TYPE_BOOLEAN,
                                     G_TYPE_INVALID);
}

static void
mozhelper_history_ac_result_received_cb (DBusGProxy *proxy,
                                      guint search_id,
                                      const gchar *value,
                                      const gchar *comment,
                                      gpointer data)
{
  g_signal_emit (data, history_signals[AC_RESULT_RECEIVED_SIGNAL], 0,
                 search_id, value, comment);
}

static void
mozhelper_history_link_visited_cb (DBusGProxy  *proxy,
                             const gchar *uri,
                             gint         visit_time,
                             gpointer     data)
{
  g_signal_emit (data, history_signals[LINK_VISITED_SIGNAL], 0,
                 uri, visit_time);
}

static void
mozhelper_history_favorites_received_cb (DBusGProxy  *proxy,
                                   gchar **urls,
                                   gchar **titles,
                                   gpointer data)
{
  g_signal_emit (data, history_signals[FAVORITES_RECEIVED_SIGNAL], 0,
                 urls, titles);
}

static void
mozhelper_history_pinned_page_cb (DBusGProxy  *proxy,
                            gchar *title,
                            gchar *url,
                            glong visit_time,
                            gboolean more_pending,
                            gpointer data)
{
  g_signal_emit (data, history_signals[PINNED_PAGE_SIGNAL], 0,
                 title, url, visit_time, more_pending);
}

static void
mozhelper_history_unpinned_page_cb (DBusGProxy  *proxy,
                              gchar *url,
                              gpointer data)
{
  g_signal_emit (data, history_signals[UNPINNED_PAGE_SIGNAL], 0, url);
}

static void
mozhelper_history_init (MozhelperHistory *self)
{
  MozhelperHistoryPrivate *priv;
  DBusGConnection *connection;
  GError *error = NULL;

  priv = self->priv = MOZHELPER_HISTORY_GET_PRIVATE (self);

  if ((connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)) == NULL)
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
    }
  else
    {
      priv->proxy
        = dbus_g_proxy_new_for_name (connection,
                                     MOZHELPER_SERVICE_HISTORY,
                                     MOZHELPER_SERVICE_HISTORY_PATH,
                                     MOZHELPER_SERVICE_HISTORY_INTERFACE);

      dbus_g_connection_unref (connection);

      dbus_g_proxy_add_signal (priv->proxy, "AcResultReceived",
                               G_TYPE_UINT,
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "LinkVisited",
                               G_TYPE_STRING,
                               G_TYPE_INT,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "FavoritesReceived",
                               G_TYPE_STRV,
                               G_TYPE_STRV,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "PinnedPage",
                               G_TYPE_STRING,
                               G_TYPE_STRING,
                               G_TYPE_INT,
                               G_TYPE_BOOLEAN,
                               G_TYPE_INVALID);
      dbus_g_proxy_add_signal (priv->proxy, "UnpinnedPage",
                               G_TYPE_STRING,
                               G_TYPE_INVALID);

      dbus_g_proxy_connect_signal
        (priv->proxy, "AcResultReceived",
         G_CALLBACK (mozhelper_history_ac_result_received_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "LinkVisited",
         G_CALLBACK (mozhelper_history_link_visited_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "FavoritesReceived",
         G_CALLBACK (mozhelper_history_favorites_received_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "PinnedPage",
         G_CALLBACK (mozhelper_history_pinned_page_cb),
         self, NULL);
      dbus_g_proxy_connect_signal
        (priv->proxy, "UnpinnedPage",
         G_CALLBACK (mozhelper_history_unpinned_page_cb),
         self, NULL);
    }
}

static void
mozhelper_history_dispose (GObject *object)
{
  MozhelperHistory *self = (MozhelperHistory *) object;
  MozhelperHistoryPrivate *priv = self->priv;

  if (priv->proxy)
    {
      while (priv->callbacks)
        {
          MozhelperHistoryClosure *closure
            = priv->callbacks->data;
          /* This should also destroy the closure and remove it from
             the list */
          dbus_g_proxy_cancel_call (priv->proxy, closure->proxy_call);
        }

      g_object_unref (priv->proxy);
      priv->proxy = NULL;
    }

  G_OBJECT_CLASS (mozhelper_history_parent_class)->dispose (object);
}

MozhelperHistory *
mozhelper_history_new (void)
{
  MozhelperHistory *self = (MozhelperHistory *) g_object_new (MOZHELPER_TYPE_HISTORY,
                                                        NULL);

  return self;
}

static gboolean
mozhelper_history_check_proxy (MozhelperHistory *self,
                            GError **error)
{
  if (self->priv->proxy)
    return TRUE;

  g_set_error (error, MOZHELPER_ERROR,
               MOZHELPER_ERROR_PROXY,
               "Failed to initialize DBUS proxy");

  return FALSE;
}

gboolean
mozhelper_history_add_uri (MozhelperHistory *self,
                        const gchar *uri,
                        gboolean redirect,
                        gboolean top_level,
                        const gchar *referrer,
                        GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_add_uri (priv->proxy,
                                           uri,
                                           redirect,
                                           top_level,
                                           referrer,
                                           error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_set_page_title (MozhelperHistory *self,
                               const gchar *uri,
                               const gchar *title,
                               GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_set_page_title (priv->proxy,
                                                  uri, title,
                                                  error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_is_visited (MozhelperHistory *self,
                           const gchar *uri,
                           gboolean *is_visited,
                           GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_is_visited (priv->proxy,
                                              uri, is_visited,
                                              error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_start_ac_search (MozhelperHistory *self,
                                const gchar *search_str,
                                guint32 *search_id,
                                GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_start_ac_search (priv->proxy,
                                                   search_str, search_id,
                                                   error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_stop_ac_search (MozhelperHistory *self,
                               guint32 search_id,
                               GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_stop_ac_search (priv->proxy,
                                                  search_id,
                                                  error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_set_favicon_url (MozhelperHistory *self,
                                const gchar *page_uri,
                                const gchar *favicon_uri,
                                GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_set_favicon_url (priv->proxy,
                                                   page_uri, favicon_uri,
                                                   error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_set_default_favicon_url (MozhelperHistory *self,
                                        const gchar *page_uri,
                                        GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_set_default_favicon_url (priv->proxy,
                                                           page_uri,
                                                           error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

static void
mozhelper_history_closure_destroy_cb (gpointer data)
{
  MozhelperHistoryClosure *closure = data;
  MozhelperHistoryPrivate *priv = closure->history->priv;

  if (closure->user_data_notify)
    closure->user_data_notify (closure->user_data);

  priv->callbacks = g_slist_remove (priv->callbacks, closure);

  g_slice_free (MozhelperHistoryClosure, closure);
}

static void
mozhelper_history_get_favicon_cb (DBusGProxy *proxy,
                               DBusGProxyCall *proxy_call,
                               void *data)
{
  MozhelperHistoryClosure *closure = data;
  GError *error = NULL;
  gchar *mime_type = NULL;
  GArray *icon_data = NULL;

  if (dbus_g_proxy_end_call (proxy, proxy_call, &error,
                             G_TYPE_STRING, &mime_type,
                             DBUS_TYPE_G_UCHAR_ARRAY, &icon_data,
                             G_TYPE_INVALID))
    {
      ((MozhelperHistoryFaviconCallback) closure->callback)
        (closure->history,
         mime_type,
         (guint8 *) icon_data->data,
         icon_data->len,
         NULL,
         closure->user_data);

      g_free (mime_type);
      g_array_free (icon_data, TRUE);
    }
  else
    {
      _mozhelper_error_translate_from_dbus (&error);

      ((MozhelperHistoryFaviconCallback) closure->callback)
        (closure->history,
         NULL,
         NULL, 0,
         error,
         closure->user_data);

      g_error_free (error);
    }
}

guint
mozhelper_history_get_favicon (MozhelperHistory *self,
                            const gchar *page_uri,
                            gboolean download,
                            MozhelperHistoryFaviconCallback callback,
                            gpointer user_data,
                            GDestroyNotify user_data_notify)
{
  MozhelperHistoryPrivate *priv;
  MozhelperHistoryClosure *closure;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  priv = self->priv;

  if (++priv->callback_id == 0)
    priv->callback_id = 1;

  closure = g_slice_new0 (MozhelperHistoryClosure);
  closure->history = self;
  closure->id = priv->callback_id;
  closure->callback = callback;
  closure->user_data = user_data;
  closure->user_data_notify = user_data_notify;

  closure->proxy_call
    = dbus_g_proxy_begin_call (priv->proxy,
                               "GetFavicon",
                               mozhelper_history_get_favicon_cb,
                               closure,
                               mozhelper_history_closure_destroy_cb,
                               G_TYPE_STRING,
                               page_uri,
                               G_TYPE_BOOLEAN,
                               download,
                               G_TYPE_INVALID);

  priv->callbacks = g_slist_prepend (priv->callbacks, closure);

  return closure->id;
}

static void
mozhelper_history_get_favorites_cb (DBusGProxy *proxy,
                              GError *error,
                              gpointer userdata)
{
  /* We don't need to do anything. The service will fire a signal when
     it actually gets the results */
}

void
mozhelper_history_get_favorites (MozhelperHistory *self)
{
  MozhelperHistoryPrivate *priv;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mozhelper_History_get_favorites_async (priv->proxy,
                                              mozhelper_history_get_favorites_cb,
                                              self);
}

gboolean
mozhelper_history_remove_favorite (MozhelperHistory *self,
                             const gchar *uri,
                             GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_remove_favorite (priv->proxy,
                                                uri,
                                                error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

gboolean
mozhelper_history_clear_history (MozhelperHistory *self,
                           GError **error)
{
  MozhelperHistoryPrivate *priv;
  gboolean ret;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  priv = self->priv;

  if (!mozhelper_history_check_proxy (self, error))
    return FALSE;

  ret = org_moblin_mozhelper_History_clear_history (priv->proxy,
                                              error);

  _mozhelper_error_translate_from_dbus (error);

  return ret;
}

static void
mozhelper_history_get_is_page_pinned_cb (DBusGProxy *proxy,
                                   DBusGProxyCall *proxy_call,
                                   void *data)
{
  MozhelperHistoryClosure *closure = data;
  GError *error = NULL;
  gboolean is_page_pinned;

  if (dbus_g_proxy_end_call (proxy, proxy_call, &error,
                             G_TYPE_BOOLEAN, &is_page_pinned,
                             G_TYPE_INVALID))
    ((MozhelperHistoryIsPagePinnedCallback) closure->callback)
      (closure->history,
       is_page_pinned,
       NULL,
       closure->user_data);
  else
    {
      _mozhelper_error_translate_from_dbus (&error);

      ((MozhelperHistoryIsPagePinnedCallback) closure->callback)
        (closure->history,
         FALSE,
         error,
         closure->user_data);

      g_error_free (error);
    }
}

guint
mozhelper_history_get_is_page_pinned (MozhelperHistory *self,
                                const gchar *page_uri,
                                MozhelperHistoryIsPagePinnedCallback callback,
                                gpointer user_data,
                                GDestroyNotify user_data_notify)
{
  MozhelperHistoryPrivate *priv;
  MozhelperHistoryClosure *closure;

  g_return_val_if_fail (MOZHELPER_IS_HISTORY (self), FALSE);

  priv = self->priv;

  if (++priv->callback_id == 0)
    priv->callback_id = 1;

  closure = g_slice_new0 (MozhelperHistoryClosure);
  closure->history = self;
  closure->id = priv->callback_id;
  closure->callback = callback;
  closure->user_data = user_data;
  closure->user_data_notify = user_data_notify;

  closure->proxy_call
    = dbus_g_proxy_begin_call (priv->proxy,
                               "GetIsPagePinned",
                               mozhelper_history_get_is_page_pinned_cb,
                               closure,
                               mozhelper_history_closure_destroy_cb,
                               G_TYPE_STRING,
                               page_uri,
                               G_TYPE_INVALID);

  priv->callbacks = g_slist_prepend (priv->callbacks, closure);

  return closure->id;
}

static void
mozhelper_history_get_pinned_pages_cb (DBusGProxy *proxy,
                                 GError *error,
                                 gpointer userdata)
{
}

void
mozhelper_history_get_pinned_pages (MozhelperHistory *self)
{
  MozhelperHistoryPrivate *priv;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mozhelper_History_get_pinned_pages_async
    (priv->proxy,
     mozhelper_history_get_pinned_pages_cb,
     self);
}

static void
mozhelper_history_pin_page_cb (DBusGProxy *proxy,
                         GError *error,
                         gpointer userdata)
{
}

void
mozhelper_history_pin_page (MozhelperHistory *self,
                      const gchar *uri,
                      const gchar *title)
{
  MozhelperHistoryPrivate *priv;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mozhelper_History_pin_page_async (priv->proxy,
                                         uri, title,
                                         mozhelper_history_pin_page_cb,
                                         self);
}

static void
mozhelper_history_unpin_page_cb (DBusGProxy *proxy,
                           GError *error,
                           gpointer userdata)
{
}

void
mozhelper_history_unpin_page (MozhelperHistory *self,
                        const gchar *uri)
{
  MozhelperHistoryPrivate *priv;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mozhelper_History_unpin_page_async (priv->proxy,
                                           uri,
                                           mozhelper_history_unpin_page_cb,
                                           self);
}

static void
mozhelper_history_unpin_all_pages_cb (DBusGProxy *proxy,
                                GError *error,
                                gpointer userdata)
{
}

void
mozhelper_history_unpin_all_pages (MozhelperHistory *self)
{
  MozhelperHistoryPrivate *priv;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  org_moblin_mozhelper_History_unpin_all_pages_async (priv->proxy,
                                                mozhelper_history_unpin_all_pages_cb,
                                                self);
}

void
mozhelper_history_cancel (MozhelperHistory *self,
                       guint id)
{
  MozhelperHistoryPrivate *priv;
  MozhelperHistoryClosure *closure;
  GSList *l;

  g_return_if_fail (MOZHELPER_IS_HISTORY (self));

  priv = self->priv;

  for (l = priv->callbacks; l; l = l->next)
    {
      closure = l->data;

      if (closure->id == id)
        {
          dbus_g_proxy_cancel_call (priv->proxy,
                                    closure->proxy_call);
          break;
        }
    }
}
