/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

Components.utils.import("resource://gre/modules/XPCOMUtils.jsm");

const Cc = Components.classes;
const Ci = Components.interfaces;

const THUMBNAIL_WIDTH    = 223;
const THUMBNAIL_HEIGHT   = 111;

/* Maximum age in milliseconds before a thumbnail is considered too
   old and will be overwritten with a new snapshot */
const MAX_THUMBNAIL_AGE  = 2 * 60 * 60 * 1000; /* two hours */

const XHTML_NS = "http://www.w3.org/1999/xhtml";

/* If no better thumbnail position can be found this will be used */
const DEFAULT_THUMBNAIL_POS = { left : 10, top : 0, note : "default" };

/* Minimum size before an image is considered a good candidate for the
 * thumbnail position */
const MIN_IMG_WIDTH = 50;
const MIN_IMG_HEIGHT = 20;

/* Set to true to get a red box drawn where the thumbnail would be
 * grabbed */
const DEBUG_THUMBNAIL_POS = false;

function MwbThumbnailer()
{
  /* Timer used to queue a grab */
  this._timer = null;
  /* Reference to the toplevel window that we need to grab */
  this._window = null;
  /* The directories to store thumbnails in */
  this._normalThumbnailDir = null;
  this._largeThumbnailDir = null;
}

MwbThumbnailer.prototype =
{
  classDescription : "MwbThumbnailer",
  contractID : "@moblin.org/mwb-thumbnailer;1",
  classID : Components.ID("{cf2292c8-659d-4dcc-851f-838da313b0fd}"),
  QueryInterface : XPCOMUtils.generateQI([Ci.nsIObserver,
                                          Ci.nsIWebProgressListener,
                                          Ci.nsISupportsWeakReference,
                                          Ci.nsITimerCallback]),

  _xpcom_categories : [{ category : "app-startup", service : true }],

  observe : function(subject, topic, data)
  {
    if (topic == "app-startup")
      {
        var obService = (Cc["@mozilla.org/observer-service;1"].
                         getService(Ci.nsIObserverService));
        obService.addObserver(this, "webnavigation-create", false);

        // Get the user's home directory
        var dirService
        = (Components.classes["@mozilla.org/file/directory_service;1"]
           .getService(Components.interfaces.nsIProperties));
        var homeDir = dirService.get("Home", Components.interfaces.nsIFile);

        this._normalThumbnailDir = homeDir.clone();
        this._normalThumbnailDir.append(".thumbnails");
        this._largeThumbnailDir = this._normalThumbnailDir.clone();
        this._normalThumbnailDir.append("normal");
        this._largeThumbnailDir.append("large");
      }
    else if (topic == "webnavigation-create")
      {
        var webProgress = subject.QueryInterface(Ci.nsIWebProgress);
        webProgress.addProgressListener(this,
                                        Ci.nsIWebProgress.NOTIFY_STATE_WINDOW);
      }
  },

  onStateChange : function(webProgress, request, stateFlags, status)
  {
    var checkStart = (Ci.nsIWebProgressListener.STATE_IS_WINDOW |
                      Ci.nsIWebProgressListener.STATE_START);
    var checkStop = (Ci.nsIWebProgressListener.STATE_IS_WINDOW |
                     Ci.nsIWebProgressListener.STATE_STOP);

    /* Clear any queued thumbnail if a new request is started */
    if ((stateFlags & checkStart) == checkStart)
      this.clearGrab();
    /* If the document has stopped loading with a successful status then
     * queue a snapshot */
    else if ((stateFlags & checkStop) == checkStop && status == 0)
      {
        if (!this._timer)
          {
            /* Grab a thumbnail after half of a second to give the
             * page a chance to render */
            this._timer = (Cc["@mozilla.org/timer;1"].
                           createInstance(Ci.nsITimer));
            this._timer.initWithCallback(this, 500,
                                         Ci.nsITimer.TYPE_ONE_SHOT);

            /* Keep a reference to the toplevel window so we know what
             * to grab from later */
            this._window = webProgress.DOMWindow.top;
          }
      }
  },

  onProgressChange : function(webProgress, request, curSelfProgress,
                              maxSelfProgress, curTotalProgress,
                              maxTotalProgress)
  {
  },

  onLocationChange : function(webProgress, request, location)
  {
  },

  onStatusChange : function(webProgress, request, status, message)
  {
  },

  onSecurityChange : function(webProgress, request, state)
  {
  },

  checkThumbnailAge : function(file)
  {
    if (!file.exists())
      return true;

    /* Regenerate the thumbnail if it is too old */
    return file.lastModifiedTime < Date.now() - MAX_THUMBNAIL_AGE;
  },

  getElementPagePosition : function(elem)
  {
    var pos = { left: 0, top: 0 };

    while (elem)
      {
        pos.left += elem.offsetLeft;
        pos.top += elem.offsetTop;

        if (elem.offsetParent == elem)
          break;
        else
          elem = elem.offsetParent;
      }

    return pos;
  },

  /* Try to guess a good position to grab the thumbnail. Returns an
   * object containing a position and a string for debugging
   * describing why the position was chosen */
  getThumbnailPosition : function()
  {
    var doc = this._window.document;

    /* Try to make sure the document is some variant of HTML */
    if (!doc.body)
      return DEFAULT_THUMBNAIL_POS;

    /* Array of potential elements */
    var candidates = [];

    /* An h1 is probably a good candidate for the title */
    var h1;
    if ((h1 = doc.getElementsByTagName("h1")[0]))
      candidates.push(h1);

    /* Look for images that are of a reasonable size in the top left
     * corner of the image */
    if (doc.images)
      {
        var i;

        for (i = 0; i < doc.images.length; i++)
          {
            var pos = this.getElementPagePosition(doc.images[i]);
            if (doc.images[i].width >= MIN_IMG_WIDTH &&
                doc.images[i].height >= MIN_IMG_HEIGHT &&
                pos.left < this._window.innerWidth / 2 &&
                pos.top < this._window.innerHeight / 2)
              candidates.push(doc.images[i]);
          }
      }

    if (candidates.length > 0)
      {
        /* Use the element closest to the top left */
        var bestElem;
        var bestDistance = Number.MAX_VALUE;
        var i;

        for (i = 0; i < candidates.length; i++)
          {
            var pos = this.getElementPagePosition(candidates[i]);
            if (pos.left * pos.left + pos.top * pos.top < bestDistance)
              {
                bestElem = candidates[i];
                bestDistance = pos.left * pos.left + pos.top * pos.top;
              }
          }

        if (bestElem)
          {
            var pos = this.getElementPagePosition(bestElem);
            pos.note = bestElem.tagName;
            return pos;
          }
      }
    else
      return DEFAULT_THUMBNAIL_POS;
  },

  saveThumbnail : function(file, scale)
  {
    /* Try to guess a good position to grab the thumbnail from */
    var thumbPos = this.getThumbnailPosition();

    /* Make sure the thumbnail directory exists */
    if (!file.parent.exists())
      file.parent.create(Ci.nsIFile.DIRECTORY_TYPE, 0777);

    /* Create a canvas to copy the window into */
    var canvas = this._window.document.createElementNS(XHTML_NS, "canvas");
    canvas.width = THUMBNAIL_WIDTH * scale;
    canvas.height = THUMBNAIL_HEIGHT * scale;
    var cr = canvas.getContext("2d");
    cr.scale(scale, scale);

    /* Draw the window onto the canvas */
    cr.drawWindow(this._window, thumbPos.left, thumbPos.top,
                  THUMBNAIL_WIDTH, THUMBNAIL_HEIGHT,
                  "rgba(255, 255, 255, 255)");
    /* Get a channel for the contents as a PNG via a data URL */
    var channel = (Cc["@mozilla.org/network/io-service;1"].
                   getService(Ci.nsIIOService).
                   newChannel(canvas.toDataURL(), null, null));
    /* Save the image to the file */
    var stream = (Cc["@mozilla.org/network/file-output-stream;1"].
                  createInstance(Components.interfaces.nsIFileOutputStream));
    stream.init(file,
                0x02 | /* PR_WRONLY */
                0x08 | /* PR_CREATE_FILE */
                0x20,  /* PR_TRUNCATE */
                0666,
                0);
    /* When mozheadless updates to the latest mozilla we can replace
     * this with NetUtil.asyncCopy but the current version has syntax
     * errors so it doesn't work */
    var bufferedStream = (Cc["@mozilla.org/network/buffered-output-stream;1"].
                          createInstance(Ci.nsIBufferedOutputStream));
    bufferedStream.init(stream, 8192);
    var copier = (Cc["@mozilla.org/network/async-stream-copier;1"].
                  createInstance(Ci.nsIAsyncStreamCopier));
    copier.init(channel.open(), bufferedStream, null, false, true,
                8192, true, true);
    copier.asyncCopy(null, null);
  },

  grabThumbnail : function()
  {
    this.clearGrab();

    /* DEBUG: draw a rectangle where the thumbnail is grabbed from */
    if (DEBUG_THUMBNAIL_POS)
      {
        var thumbPos = this.getThumbnailPosition();
        var doc = this._window.document;
        if (doc.body)
          {
            var div = doc.createElementNS(XHTML_NS, "div");
            div.appendChild(doc.createTextNode(thumbPos.note));
            div.style.position = "absolute";
            div.style.left = thumbPos.left + "px";
            div.style.top = thumbPos.top + "px";
            div.style.width = THUMBNAIL_WIDTH + "px";
            div.style.height = THUMBNAIL_HEIGHT + "px";
            div.style.borderColor = "red";
            div.style.borderWidth = "3px";
            div.style.borderStyle = "dashed";
            div.style.color = "red";
            doc.body.appendChild(div);
          }
      }

    /* Get the URL currently being displayed so we can generate
     * filenames for the thumbnail */
    var win = this._window.QueryInterface(Ci.nsIDOMWindowInternal);
    var url = win.location.href;

    /* Convert the string to a UTF-8 byte array */
    var utf8Converter = (Cc["@mozilla.org/intl/scriptableunicodeconverter"].
                         getService(Ci.nsIScriptableUnicodeConverter));
    utf8Converter.charset = "UTF-8";
    var urlUtf8 = utf8Converter.convertToByteArray(url, {});

    /* Get an MD5 sum of the URL */
    var cryptoHash = (Cc["@mozilla.org/security/hash;1"].
                      getService(Ci.nsICryptoHash));
    cryptoHash.init(cryptoHash.MD5);
    cryptoHash.update(urlUtf8, urlUtf8.length);
    var binHash = cryptoHash.finish(false);

    /* Convert it to ascii hex */
    var strHash = "";
    for (i = 0; i < binHash.length; i++)
      {
        if (binHash.charCodeAt(i) < 16)
          strHash += "0";
        strHash += binHash.charCodeAt(i).toString(16);
      }
    /* Build filenames of the thumbnail using the hash */
    var normalThumbnailFile = this._normalThumbnailDir.clone();
    normalThumbnailFile.append(strHash + ".png");
    var largeThumbnailFile = this._largeThumbnailDir.clone();
    largeThumbnailFile.append(strHash + ".png");

    /* Don't create the thumbnail if we already have a thumbnail that
     * is fairly new */
    if (this.checkThumbnailAge(normalThumbnailFile))
      this.saveThumbnail(normalThumbnailFile, 0.5);
    if (this.checkThumbnailAge(largeThumbnailFile))
      this.saveThumbnail(largeThumbnailFile, 1.0);
  },

  clearGrab : function()
  {
    /* Clear any queued grab */
    if (this._timer)
      {
        this._timer.cancel();
        this._timer = null;
      }
  },

  notify : function(timer)
  {
    this.grabThumbnail();
  }
};

function NSGetModule(compMgr, fileSpec)
{
  return XPCOMUtils.generateModule([MwbThumbnailer]);
}
