/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-download-manager.h"
#include "mwb-download-model.h"
#include "mwb-download.h"
#include "mwb-utils.h"
#include <moz-headless.h>
#include <glib/gi18n.h>
#include <string.h>

G_DEFINE_TYPE (MwbDownloadManager, mwb_download_manager, NBTK_TYPE_WIDGET)

#define DOWNLOAD_MANAGER_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_DOWNLOAD_MANAGER, MwbDownloadManagerPrivate))

enum
{
  DOWNLOAD_ADDED,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

struct _MwbDownloadManagerPrivate
{
  NbtkWidget *top_layout;
  NbtkWidget *scroll_view;
  NbtkWidget *list_view;
  NbtkWidget *entry;
  GList      *downloads;
};

static void
mwb_download_manager_get_property (GObject *object, guint property_id,
                                   GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_manager_set_property (GObject *object, guint property_id,
                                   const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_download_manager_dispose (GObject *object)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (object)->priv;

  if (priv->top_layout)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->top_layout));
      priv->top_layout = NULL;
    }

  if (priv->scroll_view)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->scroll_view));
      priv->scroll_view = NULL;
    }

  while (priv->downloads)
    {
      g_object_unref (G_OBJECT (priv->downloads->data));
      priv->downloads = g_list_delete_link (priv->downloads, priv->downloads);
    }

  G_OBJECT_CLASS (mwb_download_manager_parent_class)->dispose (object);
}

static void
mwb_download_manager_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_download_manager_parent_class)->finalize (object);
}

static void
mwb_download_manager_paint (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  /* Chain up to get background */
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->top_layout));
  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_pick (ClutterActor *actor, const ClutterColor *color)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  clutter_actor_paint (CLUTTER_ACTOR (priv->top_layout));
  clutter_actor_paint (CLUTTER_ACTOR (priv->scroll_view));
}

static void
mwb_download_manager_allocate (ClutterActor           *actor,
                               const ClutterActorBox  *box,
                               ClutterAllocationFlags  flags)
{
  NbtkPadding padding;
  ClutterActorBox child_box;

  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->
    allocate (actor, box, flags);

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  child_box.x1 = padding.left;
  child_box.x2 = box->x2 - box->x1 - padding.right;

  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->top_layout),
                                      child_box.x2 - child_box.x1,
                                      NULL,
                                      &child_box.y2);
  child_box.y1 = padding.top;
  child_box.y2 += child_box.y1;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->top_layout), &child_box, flags);

  child_box.y1 = child_box.y2;
  child_box.y2 = box->y2 - box->y1 - padding.bottom;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->scroll_view), &child_box, flags);
}

static void
mwb_download_manager_map (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->map (actor);
  clutter_actor_map (CLUTTER_ACTOR (priv->scroll_view));
  clutter_actor_map (CLUTTER_ACTOR (priv->top_layout));
}

static void
mwb_download_manager_unmap (ClutterActor *actor)
{
  MwbDownloadManagerPrivate *priv = MWB_DOWNLOAD_MANAGER (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_download_manager_parent_class)->unmap (actor);
  clutter_actor_unmap (CLUTTER_ACTOR (priv->scroll_view));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->top_layout));
}

static void
mwb_download_manager_class_init (MwbDownloadManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbDownloadManagerPrivate));

  object_class->get_property = mwb_download_manager_get_property;
  object_class->set_property = mwb_download_manager_set_property;
  object_class->dispose = mwb_download_manager_dispose;
  object_class->finalize = mwb_download_manager_finalize;

  actor_class->paint = mwb_download_manager_paint;
  actor_class->pick = mwb_download_manager_pick;
  actor_class->allocate = mwb_download_manager_allocate;
  actor_class->map = mwb_download_manager_map;
  actor_class->unmap = mwb_download_manager_unmap;

  signals[DOWNLOAD_ADDED] =
    g_signal_new ("download-added",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbDownloadManagerClass, download_added),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__VOID,
                  G_TYPE_NONE, 0);
}

static gboolean
mwb_download_manager_filter_cb (ClutterModel     *model,
                                ClutterModelIter *iter,
                                gpointer          user_data)
{
  gboolean returnval;

  gchar *src = NULL;
  NbtkEntry *entry = NBTK_ENTRY (user_data);
  const gchar *string = nbtk_entry_get_text (entry);

  if (!string || !(*string))
    return TRUE;

  clutter_model_iter_get (iter,
                          MWB_COL_SRC, &src,
                          -1);

  if (src && string && strcasestr (src, string))
    returnval = TRUE;
  else
    returnval = FALSE;

  g_free (src);

  return returnval;
}

static void
mwb_download_manager_text_changed_cb (ClutterText        *text,
                                      MwbDownloadManager *self)
{
  const gchar *string;
  ClutterModel *model;
  MwbDownloadManagerPrivate *priv = self->priv;

  string = nbtk_entry_get_text (NBTK_ENTRY (priv->entry));

  model = nbtk_list_view_get_model (NBTK_LIST_VIEW (priv->list_view));

  if (!string || !(*string))
    clutter_model_set_filter (model,
                              NULL,
                              NULL,
                              NULL);
  else
    clutter_model_set_filter (model,
                              mwb_download_manager_filter_cb,
                              priv->entry,
                              NULL);
}

static void
mwb_download_manager_clear_clicked_cb (NbtkButton         *button,
                                       MwbDownloadManager *self)
{
  ClutterModel *model;
  MwbDownloadManagerPrivate *priv = self->priv;

  model = nbtk_list_view_get_model (NBTK_LIST_VIEW (priv->list_view));

  mwb_download_model_clear_finished (MWB_DOWNLOAD_MODEL (model));
}

static void
mwb_download_manager_init (MwbDownloadManager *self)
{
  ClutterActor *text;
  ClutterModel *model;
  NbtkWidget *label, *button;

  MwbDownloadManagerPrivate *priv = self->priv =
    DOWNLOAD_MANAGER_PRIVATE (self);

  /* Create search box */
  priv->top_layout = nbtk_box_layout_new ();
  nbtk_box_layout_set_spacing (NBTK_BOX_LAYOUT (priv->top_layout), 32);
  clutter_actor_set_name (CLUTTER_ACTOR (priv->top_layout), "download-search");

  label = nbtk_label_new (_("Downloads:"));
  priv->entry = nbtk_entry_new ("");
  nbtk_entry_set_hint_text (NBTK_ENTRY (priv->entry), _("Type name to filter"));
  button = nbtk_button_new_with_label (_("Clear old downloads"));

  /* Connect up signals */
  text = nbtk_entry_get_clutter_text (NBTK_ENTRY (priv->entry));
  g_signal_connect (text, "text-changed",
                    G_CALLBACK (mwb_download_manager_text_changed_cb), self);
  g_signal_connect (button, "clicked",
                    G_CALLBACK (mwb_download_manager_clear_clicked_cb), self);

  clutter_container_add (CLUTTER_CONTAINER (priv->top_layout),
                         CLUTTER_ACTOR (label),
                         CLUTTER_ACTOR (priv->entry),
                         CLUTTER_ACTOR (button),
                         NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (priv->top_layout),
                               CLUTTER_ACTOR (label),
                               "y-fill", FALSE,
                               "x-fill", FALSE,
                               "x-align", NBTK_ALIGN_START,
                               NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (priv->top_layout),
                               CLUTTER_ACTOR (priv->entry),
                               "expand", TRUE,
                               NULL);
  clutter_container_child_set (CLUTTER_CONTAINER (priv->top_layout),
                               CLUTTER_ACTOR (button),
                               "y-fill", FALSE,
                               "x-fill", FALSE,
                               "x-align", NBTK_ALIGN_END,
                               NULL);

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->top_layout),
                            CLUTTER_ACTOR (self));

  /* Create downloads list */
  priv->scroll_view = nbtk_scroll_view_new ();
  priv->list_view = nbtk_list_view_new ();

  model = mwb_download_model_new ();
  nbtk_list_view_set_model (NBTK_LIST_VIEW (priv->list_view), model);
  g_object_unref (G_OBJECT (model));

  nbtk_list_view_set_item_type (NBTK_LIST_VIEW (priv->list_view),
                                MWB_TYPE_DOWNLOAD);

  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "download",
                                MWB_COL_DOWNLOAD);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "source",
                                MWB_COL_SRC);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "destination",
                                MWB_COL_DEST);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "progress",
                                MWB_COL_PROGRESS);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "max-progress",
                                MWB_COL_MAX_PROGRESS);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "completed",
                                MWB_COL_COMPLETED);
  nbtk_list_view_add_attribute (NBTK_LIST_VIEW (priv->list_view),
                                "cancelled",
                                MWB_COL_CANCELLED);

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->scroll_view),
                            CLUTTER_ACTOR (self));

  clutter_container_add_actor (CLUTTER_CONTAINER (priv->scroll_view),
                               CLUTTER_ACTOR (priv->list_view));
}

NbtkWidget *
mwb_download_manager_new (void)
{
  return g_object_new (MWB_TYPE_DOWNLOAD_MANAGER, NULL);
}

void
mwb_download_manager_add (MwbDownloadManager      *dlman,
                          ClutterMozEmbedDownload *download)
{
  MwbDownloadManagerPrivate *priv = dlman->priv;
  ClutterModel *model =
    nbtk_list_view_get_model (NBTK_LIST_VIEW (priv->list_view));

  mwb_download_model_add (MWB_DOWNLOAD_MODEL (model), download);
  g_signal_emit (dlman, signals[DOWNLOAD_ADDED], 0);
}

MwbDownloadModel *
mwb_download_manager_get_model (MwbDownloadManager *dlman)
{
  return MWB_DOWNLOAD_MODEL (
    nbtk_list_view_get_model (NBTK_LIST_VIEW (dlman->priv->list_view)));
}

