/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mwb-status-bar.h"

G_DEFINE_TYPE (MwbStatusBar, mwb_status_bar, NBTK_TYPE_WIDGET)

#define STATUS_BAR_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_STATUS_BAR, MwbStatusBarPrivate))

enum
{
  PROP_0,

  PROP_TEXT,
};

struct _MwbStatusBarPrivate
{
  NbtkWidget            *label;
  gulong                 hide_timeout;
  gchar                 *text;
};

static void
mwb_status_bar_get_property (GObject *object, guint property_id,
                             GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    case PROP_TEXT:
      g_value_set_string (value,
                          mwb_status_bar_get_text (MWB_STATUS_BAR (object)));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_status_bar_set_property (GObject *object, guint property_id,
                              const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    case PROP_TEXT:
      mwb_status_bar_set_text (MWB_STATUS_BAR (object),
                               g_value_get_string (value), TRUE);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_status_bar_dispose (GObject *object)
{
  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (object)->priv;

  if (priv->hide_timeout)
    {
      g_source_remove (priv->hide_timeout);
      priv->hide_timeout = 0;
    }

  if (priv->label)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->label));
      priv->label = NULL;
    }

  G_OBJECT_CLASS (mwb_status_bar_parent_class)->dispose (object);
}

static void
mwb_status_bar_finalize (GObject *object)
{
  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (object)->priv;

  g_free (priv->text);

  G_OBJECT_CLASS (mwb_status_bar_parent_class)->finalize (object);
}

static void
mwb_status_bar_get_preferred_width (ClutterActor *actor,
                                    gfloat        for_height,
                                    gfloat       *min_width_p,
                                    gfloat       *nat_width_p)
{
  NbtkPadding padding;

  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  clutter_actor_get_preferred_width (CLUTTER_ACTOR (priv->label),
                                     for_height,
                                     min_width_p,
                                     nat_width_p);

  if (min_width_p)
    *min_width_p += padding.left + padding.right;

  if (nat_width_p)
    *nat_width_p += padding.left + padding.right;
}

static void
mwb_status_bar_get_preferred_height (ClutterActor *actor,
                                     gfloat        for_width,
                                     gfloat       *min_height_p,
                                     gfloat       *nat_height_p)
{
  NbtkPadding padding;

  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->label),
                                      for_width,
                                      min_height_p,
                                      nat_height_p);

  if (min_height_p)
    *min_height_p += padding.top + padding.bottom;

  if (nat_height_p)
    *nat_height_p += padding.top + padding.bottom;
}

static void
mwb_status_bar_paint (ClutterActor *actor)
{
  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  /* Chain up to get background */
  CLUTTER_ACTOR_CLASS (mwb_status_bar_parent_class)->paint (actor);

  clutter_actor_paint (CLUTTER_ACTOR (priv->label));
}

static void
mwb_status_bar_allocate (ClutterActor           *actor,
                         const ClutterActorBox  *box,
                         ClutterAllocationFlags  flags)
{
  NbtkPadding padding;
  ClutterActorBox child_box;

  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_status_bar_parent_class)->allocate (actor, box, flags);

  nbtk_widget_get_padding (NBTK_WIDGET (actor), &padding);

  child_box = (ClutterActorBox)
              { padding.left,
                padding.top,
                box->x2 - box->x1 - padding.right,
                box->y2 - box->y1 - padding.bottom };
  clutter_actor_allocate (CLUTTER_ACTOR (priv->label), &child_box, flags);
}

static void
mwb_status_bar_map (ClutterActor *actor)
{
  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_status_bar_parent_class)->map (actor);

  clutter_actor_map (CLUTTER_ACTOR (priv->label));
}

static void
mwb_status_bar_unmap (ClutterActor *actor)
{
  MwbStatusBarPrivate *priv = MWB_STATUS_BAR (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_status_bar_parent_class)->unmap (actor);

  clutter_actor_unmap (CLUTTER_ACTOR (priv->label));
}

static void
mwb_status_bar_class_init (MwbStatusBarClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbStatusBarPrivate));

  object_class->get_property = mwb_status_bar_get_property;
  object_class->set_property = mwb_status_bar_set_property;
  object_class->dispose = mwb_status_bar_dispose;
  object_class->finalize = mwb_status_bar_finalize;

  actor_class->get_preferred_width = mwb_status_bar_get_preferred_width;
  actor_class->get_preferred_height = mwb_status_bar_get_preferred_height;
  actor_class->paint = mwb_status_bar_paint;
  actor_class->allocate = mwb_status_bar_allocate;
  actor_class->map = mwb_status_bar_map;
  actor_class->unmap = mwb_status_bar_unmap;

  g_object_class_install_property (object_class,
                                   PROP_TEXT,
                                   g_param_spec_string ("text",
                                                        "Text",
                                                        "Status text.",
                                                        "",
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));
}

static void
mwb_status_bar_init (MwbStatusBar *self)
{
  MwbStatusBarPrivate *priv = self->priv = STATUS_BAR_PRIVATE (self);

  priv->label = nbtk_label_new ("");
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->label), CLUTTER_ACTOR (self));

  priv->text = g_strdup ("");

  clutter_actor_set_opacity (CLUTTER_ACTOR (self), 0x00);
}

NbtkWidget *
mwb_status_bar_new (void)
{
  return NBTK_WIDGET (g_object_new (MWB_TYPE_STATUS_BAR,
                                    "show-on-set-parent", FALSE, NULL));
}

static void
mwb_status_bar_fadeout_completed_cb (ClutterAnimation *animation,
                                     ClutterActor    *bar)
{
  if (clutter_actor_get_opacity (bar) == 0x00)
    clutter_actor_hide (bar);

  g_signal_handlers_disconnect_by_func (animation,
                                        mwb_status_bar_fadeout_completed_cb,
                                        bar);
}

static gboolean
mwb_status_bar_hide_timeout_cb (MwbStatusBar *bar)
{
  ClutterAnimation *animation;

  MwbStatusBarPrivate *priv = bar->priv;

  animation = clutter_actor_animate (CLUTTER_ACTOR (bar),
                                     CLUTTER_EASE_IN_SINE,
                                     150,
                                     "opacity", 0x00,
                                     "signal::completed",
                                       mwb_status_bar_fadeout_completed_cb,
                                       bar,
                                     NULL);

  priv->hide_timeout = 0;

  return FALSE;
}

void
mwb_status_bar_hide (MwbStatusBar *bar)
{
  MwbStatusBarPrivate *priv = bar->priv;

  if (priv->hide_timeout)
    g_source_remove (priv->hide_timeout);

  mwb_status_bar_hide_timeout_cb (bar);
}

void
mwb_status_bar_set_text (MwbStatusBar *bar,
                         const gchar *text,
                         gboolean autohide)
{
  MwbStatusBarPrivate *priv = bar->priv;

  if (!text || text[0] == '\0')
    {
      mwb_status_bar_hide (bar);
      priv->text[0] = '\0';
      return;
    }

  nbtk_label_set_text (NBTK_LABEL (priv->label), text);
  g_free (priv->text);
  priv->text = g_strdup (text);

  if (priv->hide_timeout)
    {
      g_source_remove (priv->hide_timeout);
      priv->hide_timeout = 0;
    }

  clutter_actor_show (CLUTTER_ACTOR (bar));
  clutter_actor_animate (CLUTTER_ACTOR (bar),
                         CLUTTER_EASE_IN_SINE,
                         150,
                         "opacity", 0xff,
                         NULL);

  if (autohide)
    {
      priv->hide_timeout =
        g_timeout_add_seconds (5,
                               (GSourceFunc)mwb_status_bar_hide_timeout_cb,
                               bar);
    }
}

const gchar *
mwb_status_bar_get_text (MwbStatusBar *bar)
{
  MwbStatusBarPrivate *priv = bar->priv;
  return priv->text;
}
