/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n.h>

#include "mwb-mozembed.h"
#include "mwb-popup.h"

static void scrollable_interface_init (NbtkScrollableInterface *iface);

static void scrollable_set_adjustments (NbtkScrollable *scrollable,
                                        NbtkAdjustment *hadjustment,
                                        NbtkAdjustment *vadjustment);

static void scrollable_get_adjustments (NbtkScrollable  *scrollable,
                                        NbtkAdjustment **hadjustment,
                                        NbtkAdjustment **vadjustment);

G_DEFINE_TYPE_WITH_CODE (MwbMozEmbed, mwb_mozembed, CLUTTER_TYPE_MOZEMBED,
                         G_IMPLEMENT_INTERFACE(NBTK_TYPE_SCROLLABLE,
                                               scrollable_interface_init))

#define MOZEMBED_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_MOZEMBED, MwbMozEmbedPrivate))

enum
{
  NEW_TAB,

  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0, };

static const char const *
mwb_mozembed_comp_paths[] =
{
  PKGDATADIR "/components",
  NULL
};

static const char const *
mwb_mozembed_chrome_paths[] =
{
  PKGDATADIR "/chrome",
  NULL
};

struct _MwbMozEmbedPrivate
{
  NbtkAdjustment *hadjust;
  NbtkAdjustment *vadjust;
  gboolean        sync_adjust;
  guint           adjust_idle;

  NbtkWidget     *tooltip_hack;
  NbtkWidget     *popup;

  gchar          *last_link_message;
  gchar          *active_link_message;
};

enum
{
  PROP_0,

  PROP_HADJUST,
  PROP_VADJUST,
  PROP_SYNC_ADJUSTMENTS
};

static void
mwb_mozembed_get_property (GObject *object, guint property_id,
                           GValue *value, GParamSpec *pspec)
{
  NbtkAdjustment *adjustment;

  switch (property_id)
    {
    case PROP_HADJUST :
      scrollable_get_adjustments (NBTK_SCROLLABLE (object), &adjustment, NULL);
      g_value_set_object (value, adjustment);
      break;

    case PROP_VADJUST :
      scrollable_get_adjustments (NBTK_SCROLLABLE (object), NULL, &adjustment);
      g_value_set_object (value, adjustment);
      break;

    case PROP_SYNC_ADJUSTMENTS :
      g_value_set_boolean (value, mwb_mozembed_get_sync_adjustments (
                                    MWB_MOZEMBED (object)));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_mozembed_set_property (GObject *object, guint property_id,
                           const GValue *value, GParamSpec *pspec)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  switch (property_id)
    {
    case PROP_HADJUST :
      scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                  g_value_get_object (value),
                                  priv->vadjust);
      break;

    case PROP_VADJUST :
      scrollable_set_adjustments (NBTK_SCROLLABLE (object),
                                  priv->hadjust,
                                  g_value_get_object (value));
      break;

    case PROP_SYNC_ADJUSTMENTS :
      mwb_mozembed_set_sync_adjustments (MWB_MOZEMBED (object),
                                         g_value_get_boolean (value));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
hadjust_value_notify_cb (NbtkAdjustment *adjustment,
                         GParamSpec     *pspec,
                         MwbMozEmbed    *mozembed)
{
  gdouble value = nbtk_adjustment_get_value (adjustment);
  g_object_set (G_OBJECT (mozembed), "scroll-x", (gint)value, NULL);
}

static void
vadjust_value_notify_cb (NbtkAdjustment *adjustment,
                         GParamSpec     *pspec,
                         MwbMozEmbed    *mozembed)
{
  gdouble value = nbtk_adjustment_get_value (adjustment);
  g_object_set (G_OBJECT (mozembed), "scroll-y", (gint)value, NULL);
}

static void
mwb_mozembed_dispose (GObject *object)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  if (priv->adjust_idle)
    {
      g_source_remove (priv->adjust_idle);
      priv->adjust_idle = 0;
    }

  if (priv->hadjust)
    {
      g_signal_handlers_disconnect_by_func (priv->hadjust,
                                            hadjust_value_notify_cb,
                                            object);
      g_object_unref (priv->hadjust);
      priv->hadjust = NULL;
    }

  if (priv->vadjust)
    {
      g_signal_handlers_disconnect_by_func (priv->vadjust,
                                            vadjust_value_notify_cb,
                                            object);
      g_object_unref (priv->vadjust);
      priv->vadjust = NULL;
    }

  if (priv->popup)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->popup));
      priv->popup = NULL;
    }

  if (priv->tooltip_hack)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->tooltip_hack));
      priv->tooltip_hack = NULL;
    }

  G_OBJECT_CLASS (mwb_mozembed_parent_class)->dispose (object);
}

static void
mwb_mozembed_finalize (GObject *object)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (object)->priv;

  g_free (priv->active_link_message);
  g_free (priv->last_link_message);

  G_OBJECT_CLASS (mwb_mozembed_parent_class)->finalize (object);
}

/* NOTE: This is really working around an Nbtk bug, but quicker
 *       to fix it here for now.
 */
static gboolean
mwb_mozembed_update_adjustments_cb (MwbMozEmbed *self)
{
  ClutterActorBox box;
  MwbMozEmbedPrivate *priv = self->priv;

  clutter_actor_get_allocation_box (CLUTTER_ACTOR (self), &box);

  if (priv->hadjust)
    g_object_set (G_OBJECT (priv->hadjust),
                  "page-size", (gdouble)(box.x2 - box.x1),
                  "page-increment", (box.x2 - box.x1)/2.0, NULL);
  if (priv->vadjust)
    g_object_set (G_OBJECT (priv->vadjust),
                  "page-size", (gdouble)(box.y2 - box.y1),
                  "page-increment", (box.y2 - box.y1)/2.0, NULL);

  priv->adjust_idle = 0;

  return FALSE;
}

static void
mwb_mozembed_allocate (ClutterActor           *actor,
                       const ClutterActorBox  *box,
                       ClutterAllocationFlags  flags)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->allocate (actor, box, flags);

  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->tooltip_hack),
                                         flags);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->popup),
                                         flags);

  if (!priv->adjust_idle)
    priv->adjust_idle =
      g_idle_add_full (CLUTTER_PRIORITY_REDRAW,
                       (GSourceFunc)mwb_mozembed_update_adjustments_cb,
                       actor,
                       NULL);

/*  if (priv->hadjust)
    g_object_set (G_OBJECT (priv->hadjust),
                  "page-size", (gdouble)(box->x2 - box->x1),
                  "page-increment", (box->x2 - box->x1)/2.0, NULL);
  if (priv->vadjust)
    g_object_set (G_OBJECT (priv->vadjust),
                  "page-size", (gdouble)(box->y2 - box->y1),
                  "page-increment", (box->y2 - box->y1)/2.0, NULL);*/
}

static void
open_link_cb (MwbAction   *action,
              MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  g_signal_emit (self, signals[NEW_TAB], 0, priv->active_link_message);
}

static void
copy_link_cb (MwbAction   *action,
              MwbMozEmbed *self)
{
  nbtk_clipboard_set_text (nbtk_clipboard_get_default (),
                           self->priv->active_link_message);
}

static gboolean
mwb_mozembed_button_press_event (ClutterActor       *actor,
                                 ClutterButtonEvent *event)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  if (mwb_popup_is_visible (MWB_POPUP (priv->popup)))
    mwb_popup_hide (MWB_POPUP (priv->popup));
  else if ((event->button == 3) && priv->last_link_message)
    {
      gfloat x, y;
      if (clutter_actor_transform_stage_point (actor,
                                               event->x,
                                               event->y,
                                               &x,
                                               &y))
        {
          g_free (priv->active_link_message);
          priv->active_link_message = g_strdup (priv->last_link_message);

          mwb_popup_clear (MWB_POPUP (priv->popup));
          mwb_popup_add_action (MWB_POPUP (priv->popup),
                                mwb_action_new_full (_("Open link in new tab"),
                                                     G_CALLBACK (open_link_cb),
                                                     actor));
          mwb_popup_add_action (MWB_POPUP (priv->popup),
                                mwb_action_new_full (_("Copy link location"),
                                                     G_CALLBACK (copy_link_cb),
                                                     actor));
          clutter_actor_set_position (CLUTTER_ACTOR (priv->popup), x, y);
          mwb_popup_show (MWB_POPUP (priv->popup));
        }
    }

  return CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->
    button_press_event (actor, event);
}

static void
mwb_mozembed_paint (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->paint (actor);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->popup))
    clutter_actor_paint (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_pick (ClutterActor       *actor,
                   const ClutterColor *color)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->pick (actor, color);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->popup))
    clutter_actor_paint (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_map (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->map (actor);
  clutter_actor_map (CLUTTER_ACTOR (priv->tooltip_hack));
  clutter_actor_map (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_unmap (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;
  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->unmap (actor);
  clutter_actor_unmap (CLUTTER_ACTOR (priv->tooltip_hack));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->popup));
}

static void
mwb_mozembed_key_focus_out (ClutterActor *actor)
{
  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (actor)->priv;

  mwb_popup_hide (MWB_POPUP (priv->popup));

  CLUTTER_ACTOR_CLASS (mwb_mozembed_parent_class)->key_focus_out (actor);
}

static void
mwb_mozembed_class_init (MwbMozEmbedClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbMozEmbedPrivate));

  object_class->get_property = mwb_mozembed_get_property;
  object_class->set_property = mwb_mozembed_set_property;
  object_class->dispose = mwb_mozembed_dispose;
  object_class->finalize = mwb_mozembed_finalize;

  actor_class->allocate = mwb_mozembed_allocate;
  actor_class->button_press_event = mwb_mozembed_button_press_event;
  actor_class->paint = mwb_mozembed_paint;
  actor_class->pick = mwb_mozembed_pick;
  actor_class->map = mwb_mozembed_map;
  actor_class->unmap = mwb_mozembed_unmap;
  actor_class->key_focus_out = mwb_mozembed_key_focus_out;

  g_object_class_install_property (object_class,
                                   PROP_SYNC_ADJUSTMENTS,
                                   g_param_spec_boolean ("sync-adjustments",
                                                         "Sync adjustments",
                                                         "Keep adjustment values "
                                                         "synchronised with "
                                                         "internal state.",
                                                         TRUE,
                                                         G_PARAM_READWRITE |
                                                         G_PARAM_STATIC_NAME |
                                                         G_PARAM_STATIC_NICK |
                                                         G_PARAM_STATIC_BLURB));

  signals[NEW_TAB] =
    g_signal_new ("new-tab",
                  G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (MwbMozEmbedClass, new_tab),
                  NULL, NULL,
                  g_cclosure_marshal_VOID__STRING,
                  G_TYPE_NONE, 1, G_TYPE_STRING);

  g_object_class_override_property (object_class,
                                    PROP_HADJUST,
                                    "hadjustment");

  g_object_class_override_property (object_class,
                                    PROP_VADJUST,
                                    "vadjustment");
}

static void
mwb_mozembed_scroll_x_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->hadjust && priv->sync_adjust)
    {
      gint scroll_x;
      g_object_get (G_OBJECT (self), "scroll-x", &scroll_x, NULL);
      nbtk_adjustment_set_value (priv->hadjust, scroll_x);
    }

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_scroll_y_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->vadjust && priv->sync_adjust)
    {
      gint scroll_y;
      g_object_get (G_OBJECT (self), "scroll-y", &scroll_y, NULL);
      nbtk_adjustment_set_value (priv->vadjust, scroll_y);
    }

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_doc_width_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->hadjust)
    {
      gint doc_width;
      g_object_get (G_OBJECT (self), "doc-width", &doc_width, NULL);
      g_object_set (G_OBJECT (priv->hadjust),
                    "upper", (gdouble)doc_width, NULL);
    }

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_doc_height_notify_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (priv->vadjust)
    {
      gint doc_height;
      g_object_get (G_OBJECT (self), "doc-height", &doc_height, NULL);
      g_object_set (G_OBJECT (priv->vadjust),
                    "upper", (gdouble)doc_height, NULL);
    }

  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_show_tooltip_cb (MwbMozEmbed *self,
                              const gchar *text,
                              gint         x,
                              gint         y)
{
  MwbMozEmbedPrivate *priv = self->priv;

  if (mwb_popup_is_visible (MWB_POPUP (priv->popup)))
    return;

  clutter_actor_set_position (CLUTTER_ACTOR (priv->tooltip_hack), x, y);
  clutter_actor_allocate_preferred_size (CLUTTER_ACTOR (priv->tooltip_hack), 0);
  nbtk_widget_set_tooltip_text (priv->tooltip_hack, text);
  nbtk_widget_show_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_hide_tooltip_cb (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv;
  nbtk_widget_hide_tooltip (priv->tooltip_hack);
}

static void
mwb_mozembed_link_message_cb (MwbMozEmbed *self,
                              const gchar *message)
{
  MwbMozEmbedPrivate *priv = self->priv;

  g_free (priv->last_link_message);
  priv->last_link_message = (message && *message) ? g_strdup (message) : NULL;
}

static void
mwb_mozembed_init (MwbMozEmbed *self)
{
  MwbMozEmbedPrivate *priv = self->priv = MOZEMBED_PRIVATE (self);

  priv->sync_adjust = TRUE;

  g_signal_connect (self, "notify::scroll-x",
                    G_CALLBACK (mwb_mozembed_scroll_x_notify_cb), NULL);
  g_signal_connect (self, "notify::scroll-y",
                    G_CALLBACK (mwb_mozembed_scroll_y_notify_cb), NULL);
  g_signal_connect (self, "notify::doc-width",
                    G_CALLBACK (mwb_mozembed_doc_width_notify_cb), NULL);
  g_signal_connect (self, "notify::doc-height",
                    G_CALLBACK (mwb_mozembed_doc_height_notify_cb), NULL);
  g_signal_connect (self, "show-tooltip",
                    G_CALLBACK (mwb_mozembed_show_tooltip_cb), NULL);
  g_signal_connect (self, "hide-tooltip",
                    G_CALLBACK (mwb_mozembed_hide_tooltip_cb), NULL);
  g_signal_connect (self, "link-message",
                    G_CALLBACK (mwb_mozembed_link_message_cb), NULL);

  priv->tooltip_hack = nbtk_bin_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->tooltip_hack),
                            CLUTTER_ACTOR (self));

  priv->popup = mwb_popup_new ();
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->popup),
                            CLUTTER_ACTOR (self));
}

ClutterActor *
mwb_mozembed_new (gboolean private)
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "scrollbars", FALSE,
                       "private", private,
                       NULL);
}

ClutterActor *
mwb_mozembed_new_with_parent (ClutterMozEmbed *parent)
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "parent", parent,
                       "scrollbars", FALSE,
                       NULL);
}

ClutterActor *
mwb_mozembed_new_for_new_window ()
{
  return g_object_new (MWB_TYPE_MOZEMBED,
                       "comp-paths", mwb_mozembed_comp_paths,
                       "chrome-paths", mwb_mozembed_chrome_paths,
                       "spawn", FALSE,
                       "scrollbars", FALSE,
                       NULL);
}

static void
scrollable_set_adjustments (NbtkScrollable *scrollable,
                            NbtkAdjustment *hadjust,
                            NbtkAdjustment *vadjust)
{
  NbtkAdjustment *old_hadjust, *old_vadjust;

  MwbMozEmbedPrivate *priv = MWB_MOZEMBED (scrollable)->priv;

  old_hadjust = priv->hadjust;
  old_vadjust = priv->vadjust;

  if (priv->hadjust)
    g_signal_handlers_disconnect_by_func (priv->hadjust,
                                          hadjust_value_notify_cb,
                                          scrollable);

  if (priv->vadjust)
    g_signal_handlers_disconnect_by_func (priv->hadjust,
                                          hadjust_value_notify_cb,
                                          scrollable);

  priv->hadjust = hadjust;
  priv->vadjust = vadjust;

  if (priv->hadjust)
    {
      g_object_ref (priv->hadjust);
      g_signal_connect (priv->hadjust, "notify::value",
                        G_CALLBACK (hadjust_value_notify_cb), scrollable);
    }

  if (priv->vadjust)
    {
      g_object_ref (priv->vadjust);
      g_signal_connect (priv->vadjust, "notify::value",
                        G_CALLBACK (vadjust_value_notify_cb), scrollable);
    }

  if (old_hadjust)
    g_object_unref (old_hadjust);
  if (old_vadjust)
    g_object_unref (old_vadjust);
}

static void
mwb_mozembed_ensure_adjustments (MwbMozEmbed *mozembed)
{
  gint doc_width, doc_height, scroll_x, scroll_y, width, height;

  MwbMozEmbedPrivate *priv = mozembed->priv;

  if (priv->hadjust && priv->vadjust)
    return;

  g_object_get (G_OBJECT (mozembed),
                "doc-width", &doc_width,
                "doc-height", &doc_height,
                "scroll-x", &scroll_x,
                "scroll-y", &scroll_y,
                NULL);
  clutter_texture_get_base_size (CLUTTER_TEXTURE (mozembed),
                                 &width, &height);

  if (!priv->hadjust)
    {
      priv->hadjust = nbtk_adjustment_new (scroll_x,
                                           0.0,
                                           doc_width,
                                           8.0,
                                           width/2.0,
                                           width);
    }

  if (!priv->vadjust)
    {
      priv->vadjust = nbtk_adjustment_new (scroll_y,
                                           0.0,
                                           doc_height,
                                           8.0,
                                           height/2.0,
                                           height);
    }

  scrollable_set_adjustments (NBTK_SCROLLABLE (mozembed),
                              priv->hadjust,
                              priv->vadjust);
}

static void
scrollable_get_adjustments (NbtkScrollable *scrollable,
                            NbtkAdjustment **hadjustment,
                            NbtkAdjustment **vadjustment)
{
  MwbMozEmbed *mozembed = MWB_MOZEMBED (scrollable);
  MwbMozEmbedPrivate *priv = mozembed->priv;

  mwb_mozembed_ensure_adjustments (mozembed);

  if (hadjustment)
    *hadjustment = priv->hadjust;

  if (vadjustment)
    *vadjustment = priv->vadjust;
}

static void
scrollable_interface_init (NbtkScrollableInterface *iface)
{
  iface->set_adjustments = scrollable_set_adjustments;
  iface->get_adjustments = scrollable_get_adjustments;
}

gboolean
mwb_mozembed_get_sync_adjustments (MwbMozEmbed *mozembed)
{
  return mozembed->priv->sync_adjust;
}

void
mwb_mozembed_set_sync_adjustments (MwbMozEmbed *mozembed, gboolean sync)
{
  MwbMozEmbedPrivate *priv = mozembed->priv;

  if (priv->sync_adjust != sync)
    {
      priv->sync_adjust = sync;

      if (sync)
        {
          gint scroll_x, scroll_y;
          g_object_get (G_OBJECT (mozembed),
                        "scroll-x", &scroll_x,
                        "scroll-y", &scroll_y,
                        NULL);
          if (priv->hadjust)
            nbtk_adjustment_set_value (priv->hadjust, scroll_x);
          if (priv->vadjust)
            nbtk_adjustment_set_value (priv->vadjust, scroll_y);
        }
    }
}

