/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <clutter/clutter.h>
#include <clutter/x11/clutter-x11.h>
#include <clutter-mozembed.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "mwb-browser.h"
#include "mwb-window.h"
#include "mwb-dbus.h"

G_DEFINE_TYPE (MwbDbus, mwb_dbus, G_TYPE_OBJECT)

#define DBUS_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_DBUS, MwbDbusPrivate))

enum
{
  PROP_0,

  PROP_BROWSER,
  PROP_WINDOW,
};

struct _MwbDbusPrivate
{
  MwbWindow  *window;
  MwbBrowser *browser;
};

#include "mwb-dbus-glue.h"

static void
mwb_dbus_get_property (GObject *object, guint property_id,
                       GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    case PROP_BROWSER:
      g_value_set_object (value, MWB_DBUS (object)->priv->browser);
      break;

    case PROP_WINDOW:
      g_value_set_object (value, MWB_DBUS (object)->priv->window);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_dbus_set_property (GObject *object, guint property_id,
                       const GValue *value, GParamSpec *pspec)
{
  switch (property_id)
    {
    case PROP_BROWSER:
      MWB_DBUS (object)->priv->browser = g_value_get_object (value);
      break;

    case PROP_WINDOW:
      MWB_DBUS (object)->priv->window = g_value_get_object (value);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_dbus_class_init (MwbDbusClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbDbusPrivate));

  object_class->get_property = mwb_dbus_get_property;
  object_class->set_property = mwb_dbus_set_property;

  g_object_class_install_property (object_class,
                                   PROP_BROWSER,
                                   g_param_spec_object ("browser",
                                                        "Browser",
                                                        "The browser actions "
                                                        "will be performed on.",
                                                        MWB_TYPE_BROWSER,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_WINDOW,
                                   g_param_spec_object ("window",
                                                        "Window",
                                                        "The window housing "
                                                        "the browser.",
                                                        MWB_TYPE_WINDOW,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  dbus_g_object_type_install_info (MWB_TYPE_DBUS,
                                   &dbus_glib_mwb_dbus_object_info);
}

static void
mwb_dbus_init (MwbDbus *self)
{
  DBusGConnection *connection;

  GError *error = NULL;
  self->priv = DBUS_PRIVATE (self);

  if (!(connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error)))
    {
      g_warning ("Error connecting to session bus: %s", error->message);
      g_error_free (error);
      return;
    }

  dbus_g_connection_register_g_object (connection,
                                       MWB_DBUS_PATH,
                                       G_OBJECT (self));
}

MwbDbus*
mwb_dbus_create (MwbWindow *window, MwbBrowser *browser)
{
  static MwbDbus *mwb_dbus = NULL;

  if (!mwb_dbus)
    {
      DBusGProxy *proxy;
      guint32 request_status;
      DBusGConnection *connection;

      GError *error = NULL;

      connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
      if (!connection)
        {
          g_warning ("Error connecting to session bus: %s", error->message);
          g_error_free (error);
          return NULL;
        }

      proxy = dbus_g_proxy_new_for_name (connection,
                                         DBUS_SERVICE_DBUS,
                                         DBUS_PATH_DBUS,
                                         DBUS_INTERFACE_DBUS);

      if (!org_freedesktop_DBus_request_name (proxy, MWB_DBUS_SERVICE,
                                              DBUS_NAME_FLAG_DO_NOT_QUEUE,
                                              &request_status, &error))
        {
          g_warning ("Failed to request name: %s", error->message);
          g_error_free (error);
          return NULL;
        }

      mwb_dbus = g_object_new (MWB_TYPE_DBUS,
                               "window", window,
                               "browser", browser,
                               NULL);
    }

  return mwb_dbus;
}

gboolean
mwb_dbus_new_tab_with_flags (MwbDbus             *self,
                             const gchar         *url,
                             MwbBrowserOpenFlags  flags,
                             GError              *error)
{
  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  mwb_browser_open (priv->browser, url, flags);

  return TRUE;
}

gboolean
mwb_dbus_new_tab    (MwbDbus      *self,
                     const gchar  *url,
                     gboolean      replace,
                     gboolean      use_existing,
                     GError       *error)
{
  MwbBrowserOpenFlags flags = MWB_BROWSER_OPEN_ACTIVATE;

  if (use_existing)
    flags |= MWB_BROWSER_OPEN_USE_EXISTING;
  if (!replace)
    flags |= MWB_BROWSER_OPEN_NEW_TAB;

  return mwb_dbus_new_tab_with_flags (self, url, flags, error);
}

gboolean
mwb_dbus_switch_tab (MwbDbus      *self,
                     gint          tab,
                     GError       *error)
{
  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  mwb_tab_picker_set_current_tab (mwb_browser_get_tab_picker (priv->browser),
                                  tab);

  return TRUE;
}

gboolean
mwb_dbus_raise      (MwbDbus      *self,
                     GError       *error)
{
  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->window)
    return TRUE;

  gtk_window_present (GTK_WINDOW (priv->window));

  return TRUE;
}

gboolean
mwb_dbus_get_tab    (MwbDbus      *self,
                     gint          tab,
                     gchar       **url,
                     gchar       **title,
                     GError       *error)
{
  GList *pages;
  ClutterMozEmbed *mozembed;

  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  pages = mwb_browser_get_pages (priv->browser);
  mozembed = g_list_nth_data (pages, tab);
  g_list_free (pages);

  if (mozembed)
    {
      if (url)
        *url = g_strdup (clutter_mozembed_get_location (mozembed));
      if (title)
        *title = g_strdup (clutter_mozembed_get_title (mozembed));
    }

  return TRUE;
}

gint
mwb_dbus_get_ntabs  (MwbDbus      *self)
{
  MwbDbusPrivate *priv = self->priv;

  if (!priv->browser)
    return 0;

  return mwb_tab_picker_get_n_tabs (mwb_browser_get_tab_picker (priv->browser))-1;
}

gint
mwb_dbus_get_current_tab (MwbDbus *self)
{
  MwbDbusPrivate *priv = self->priv;

  if (!priv->browser)
    return 0;

  return
    mwb_tab_picker_get_current_tab (mwb_browser_get_tab_picker (priv->browser));
}

gboolean
mwb_dbus_close_tab       (MwbDbus      *self,
                          gint          tab_no,
                          GError       *error)
{
  MwbTabPicker *tab_picker;
  MwbTab       *tab;

  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  tab_picker = mwb_browser_get_tab_picker (priv->browser);
  tab = mwb_tab_picker_get_tab (tab_picker, tab_no);

  if (tab && mwb_tab_get_can_close (tab))
    g_signal_emit_by_name (tab, "closed");

  return TRUE;
}

gboolean
mwb_dbus_connect_view    (MwbDbus      *self,
                          gint          tab,
                          const gchar  *input,
                          const gchar  *output,
                          GError       *error)
{
  GList *pages;
  ClutterMozEmbed *mozembed;

  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  pages = mwb_browser_get_pages (priv->browser);
  mozembed = g_list_nth_data (pages, tab);
  g_list_free (pages);

  if (mozembed)
    clutter_mozembed_connect_view (mozembed, input, output);

  return TRUE;
}

gboolean
mwb_dbus_start_private_browsing (MwbDbus      *self,
                                 GError       *error)
{
  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  mwb_browser_activate_private (priv->browser);

  return TRUE;
}

gboolean
mwb_dbus_purge_session_history (MwbDbus *self,
                                GError *error)
{
  MwbDbusPrivate *priv = self->priv;

  /* FIXME: Add an error state here */
  if (!priv->browser)
    return TRUE;

  mwb_browser_purge_session_history (priv->browser);

  return TRUE;
}
