/*
 * Moblin-Web-Browser: The web browser for Moblin
 * Copyright (c) 2009, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU Lesser General Public License,
 * version 2.1, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <clutter-mozembed.h>
#include <moz-headless.h>
#include <gtk/gtk.h>
#include <mhs/mhs.h>
#include <glib/gi18n.h>
#include <clutter-gtk/clutter-gtk.h>

#include "mwb-main.h"
#include "mwb-aspect-clone.h"
#include "mwb-browser.h"
#include "mwb-download-manager.h"
#include "mwb-mozembed.h"
#include "mwb-page.h"
#include "mwb-status-bar.h"
#include "mwb-toolbar.h"
#include "mwb-tab.h"
#include "mwb-tab-picker.h"
#include "mwb-utils.h"
#include "mwb-window.h"

G_DEFINE_TYPE (MwbBrowser, mwb_browser, CLUTTER_TYPE_ACTOR)

#define BROWSER_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), MWB_TYPE_BROWSER, MwbBrowserPrivate))

#define DEFAULT_HOME "chrome://mwbpages/content/startpage.xhtml"
#define DEFAULT_PRIVATE_HOME "chrome://mwbpages/content/startpage.xhtml?private"
#define DEFAULT_ICON PKGDATADIR "/default-favicon.png"

typedef struct _MwbBrowserTabData MwbBrowserTabData;

static void mwb_browser_tab_closed_cb (MwbTab *tab, MwbBrowser *browser);

static void mwb_browser_start_loading_favicon (MwbBrowser *browser,
                                               MwbBrowserTabData *tab_data);
static void mwb_browser_stop_loading_favicon (MwbBrowser *browser,
                                              MwbBrowserTabData *tab_data);
static MwbTab *mwb_browser_get_downloads_tab (MwbBrowser *self);
static gboolean mwb_browser_window_delete_cb (GtkWidget  *window,
                                              GdkEvent   *event,
                                              MwbBrowser *browser);

static void update_cursor (MwbBrowser *browser);

struct _MwbBrowserTabData
{
  ClutterActor *browser;
  ClutterActor *page;
  gboolean      is_pinned;
  gboolean      got_icon;
  guint         favicon_handler;
  guint         is_pinned_handler;
};

enum
{
  PROP_0,

  PROP_HOME,
  PROP_PRIVATE_HOME,
  PROP_WINDOW,
};

struct _MwbBrowserPrivate
{
  gchar                 *home;
  gchar                 *private_home;

  MwbWindow             *window;
  NbtkWidget            *toolbar;
  NbtkWidget            *tab_picker;
  NbtkWidget            *status_bar;
  NbtkWidget            *dlman;
  GList                 *pages;

  gint                   current_tab;
  ClutterActor          *current_page;

  gboolean               user_initiated_url;
  gboolean               skip_tab_transition;

  MhsHistory            *history;

  guint                  save_session_source;

  gchar                 *current_link;

  gboolean               fullscreen;
  gboolean               bar_visible;
  ClutterTimeline       *bar_hide_timeline;
  ClutterAlpha          *bar_hide_alpha;
  guint                  bar_hide_timeout;

  gboolean               use_mozembed_cursor;
  MozHeadlessCursorType  mozembed_cursor;
  guint                  page_enter_handler;
  guint                  page_leave_handler;
};

GdkCursor *cursor_cache[MOZ_HEADLESS_CURSOR_TYPE_N_TYPES];

static void
mwb_browser_get_property (GObject *object, guint property_id,
                          GValue *value, GParamSpec *pspec)
{
  MwbBrowserPrivate *priv = MWB_BROWSER (object)->priv;

  switch (property_id)
    {
    case PROP_HOME:
      g_value_set_string (value, priv->home);
      break;

    case PROP_PRIVATE_HOME:
      g_value_set_string (value, priv->private_home);
      break;

    case PROP_WINDOW:
      g_value_set_object (value, priv->window);
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
mwb_browser_toggle_toolbar (MwbBrowser *self, gboolean show)
{
  MwbBrowserPrivate *priv = self->priv;
  ClutterTimelineDirection direction =
    clutter_timeline_get_direction (priv->bar_hide_timeline);

  if (show && (direction == CLUTTER_TIMELINE_FORWARD))
    clutter_timeline_set_direction (priv->bar_hide_timeline,
                                    CLUTTER_TIMELINE_BACKWARD);
  else if (!show && (direction == CLUTTER_TIMELINE_BACKWARD))
    clutter_timeline_set_direction (priv->bar_hide_timeline,
                                    CLUTTER_TIMELINE_FORWARD);
  else if (clutter_timeline_is_playing (priv->bar_hide_timeline))
    return;

  if (!clutter_timeline_is_playing (priv->bar_hide_timeline))
    clutter_timeline_rewind (priv->bar_hide_timeline);

  clutter_timeline_start (priv->bar_hide_timeline);
}

static gboolean
mwb_browser_window_state_event_cb (GtkWidget           *window,
                                   GdkEventWindowState *event,
                                   MwbBrowser          *self)
{
  MwbBrowserPrivate *priv = self->priv;
  gboolean fullscreen;

  if (!(event->changed_mask & GDK_WINDOW_STATE_FULLSCREEN))
    return FALSE;

  fullscreen = (event->new_window_state & GDK_WINDOW_STATE_FULLSCREEN);

  if (!fullscreen)
    {
      priv->fullscreen = FALSE;
      clutter_actor_set_reactive (CLUTTER_ACTOR (self), FALSE);
    }

  mwb_browser_toggle_toolbar (self, !fullscreen);

  return FALSE;
}

static void
mwb_browser_set_property (GObject *object, guint property_id,
                          const GValue *value, GParamSpec *pspec)
{
  MwbBrowserPrivate *priv = MWB_BROWSER (object)->priv;

  switch (property_id)
    {
    case PROP_HOME:
      g_free (priv->home);
      priv->home = g_value_dup_string (value);
      break;

    case PROP_PRIVATE_HOME:
      g_free (priv->private_home);
      priv->private_home = g_value_dup_string (value);
      break;

    case PROP_WINDOW:
      if (priv->window)
        {
          g_signal_handlers_disconnect_by_func (priv->window,
                                              mwb_browser_window_state_event_cb,
                                              object);
          g_signal_handlers_disconnect_by_func (priv->window,
                                                mwb_browser_window_delete_cb,
                                                object);
        }

      priv->window = g_value_get_object (value);

      if (priv->window)
        {
          g_signal_connect (priv->window, "window-state-event",
                            G_CALLBACK (mwb_browser_window_state_event_cb),
                            object);
          g_signal_connect (priv->window, "delete-event",
                            G_CALLBACK (mwb_browser_window_delete_cb), object);
        }
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
    }
}

static void
free_cursor_cache (void)
{
  int i;
  for (i = 0; i < G_N_ELEMENTS (cursor_cache); i++)
    {
      if (cursor_cache[i])
        {
          gdk_cursor_unref (cursor_cache[i]);
          cursor_cache[i] = NULL;
        }
    }
}

static void
disconnect_page_enter_leave_handlers (MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->page_enter_handler)
    {
      g_signal_handler_disconnect (priv->current_page,
                                   priv->page_enter_handler);
      priv->page_enter_handler = 0;
    }

  if (priv->page_leave_handler)
    {
      g_signal_handler_disconnect (priv->current_page,
                                   priv->page_leave_handler);
      priv->page_leave_handler = 0;
    }
}

static void
mwb_browser_dispose (GObject *object)
{
  MwbBrowser *self = MWB_BROWSER (object);
  MwbBrowserPrivate *priv = self->priv;

  disconnect_page_enter_leave_handlers (self);

  if (priv->bar_hide_timeout)
    {
      g_source_remove (priv->bar_hide_timeout);
      priv->bar_hide_timeout = 0;
    }

  if (priv->window)
    {
      g_signal_handlers_disconnect_by_func (priv->window,
                                            mwb_browser_window_state_event_cb,
                                            object);
      priv->window = NULL;
    }

  if (priv->toolbar)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->toolbar));
      priv->toolbar = NULL;
    }

  if (priv->tab_picker)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->tab_picker));
      priv->tab_picker = NULL;
    }

  while (priv->pages)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->pages->data));
      priv->pages = g_list_delete_link (priv->pages, priv->pages);
    }

  if (priv->status_bar)
    {
      clutter_actor_unparent (CLUTTER_ACTOR (priv->status_bar));
      priv->status_bar = NULL;
    }

  if (priv->history)
    {
      g_object_unref (priv->history);
      priv->history = NULL;
    }

  if (priv->dlman)
    {
      g_object_unref (priv->dlman);
      priv->dlman = NULL;
    }

  if (priv->bar_hide_timeline)
    {
      g_object_unref (priv->bar_hide_timeline);
      g_object_unref (priv->bar_hide_alpha);
      priv->bar_hide_timeline = NULL;
      priv->bar_hide_alpha = NULL;
    }

  free_cursor_cache ();

  G_OBJECT_CLASS (mwb_browser_parent_class)->dispose (object);
}

static void
mwb_browser_finalize (GObject *object)
{
  G_OBJECT_CLASS (mwb_browser_parent_class)->finalize (object);
}

static void
mwb_browser_paint (ClutterActor *actor)
{
  GList *p;
  MwbBrowserPrivate *priv = MWB_BROWSER (actor)->priv;

  for (p = priv->pages; p; p = p->next)
    {
      if (CLUTTER_ACTOR_IS_MAPPED (p->data))
        clutter_actor_paint (CLUTTER_ACTOR (p->data));
    }

  if (CLUTTER_ACTOR_IS_MAPPED (priv->tab_picker))
    clutter_actor_paint (CLUTTER_ACTOR (priv->tab_picker));

  if (CLUTTER_ACTOR_IS_MAPPED (priv->status_bar))
    clutter_actor_paint (CLUTTER_ACTOR (priv->status_bar));

  if (CLUTTER_ACTOR_IS_MAPPED (priv->toolbar))
    clutter_actor_paint (CLUTTER_ACTOR (priv->toolbar));
}

static void
mwb_browser_pick (ClutterActor *actor, const ClutterColor *color)
{
  MwbBrowserPrivate *priv = MWB_BROWSER (actor)->priv;

  if (priv->current_page)
    clutter_actor_paint (priv->current_page);

  if (CLUTTER_ACTOR_IS_MAPPED (priv->tab_picker))
    clutter_actor_paint (CLUTTER_ACTOR (priv->tab_picker));

  if (CLUTTER_ACTOR_IS_MAPPED (priv->toolbar))
    clutter_actor_paint (CLUTTER_ACTOR (priv->toolbar));
}

static void
mwb_browser_allocate (ClutterActor           *actor,
                      const ClutterActorBox  *box,
                      ClutterAllocationFlags  flags)
{
  GList *p;
  gfloat height, bar_height, picker_height;
  NbtkPadding padding;
  ClutterActorBox child_box;

  MwbBrowserPrivate *priv = MWB_BROWSER (actor)->priv;

  child_box.x1 = 0;
  child_box.x2 = box->x2 - box->x1;

  /* Get toolbar height */
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->toolbar),
                                      child_box.x2,
                                      NULL,
                                      &bar_height);

  /* Get tab picker height */
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->tab_picker),
                                      child_box.x2,
                                      NULL,
                                      &picker_height);

  /* Allocate toolbar and tab-picker with regards to fullscreen animation */
  if (priv->bar_visible)
    child_box.y1 = 0;
  else if (clutter_timeline_is_playing (priv->bar_hide_timeline))
    child_box.y1 = -(bar_height + picker_height) *
                   clutter_alpha_get_alpha (priv->bar_hide_alpha);
  else if (priv->fullscreen)
    child_box.y1 = -(bar_height + picker_height);
  else
    child_box.y1 = 0;

  child_box.y2 = child_box.y1 + bar_height;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->toolbar), &child_box, flags);

  child_box.y1 = child_box.y2;
  child_box.y2 = child_box.y1 + picker_height;
  clutter_actor_allocate (CLUTTER_ACTOR (priv->tab_picker), &child_box, flags);

  /* Get the tab picker padding so we can negate its vertical padding when
   * allocating the main frame. The tab picker normally has no vertical padding,
   * but some is added when it switches to preview mode.
   */
  nbtk_widget_get_padding (priv->tab_picker, &padding);

  /* Allocate main frame */
  if (priv->fullscreen)
    {
      child_box.y1 = 0;
      child_box.y2 = box->y2 - box->y1;
    }
  else
    {
      mwb_tab_picker_get_preferred_height (MWB_TAB_PICKER (priv->tab_picker),
                                           child_box.x2,
                                           NULL,
                                           &height,
                                           FALSE);
      height = (box->y2 - box->y1) + ((child_box.y2 - child_box.y1) - height);
      child_box.y1 = MWB_PIXBOUND (child_box.y2);
      child_box.y2 = MWB_PIXBOUND (height) + padding.top + padding.bottom;
    }
  for (p = priv->pages; p; p = p->next)
    clutter_actor_allocate (CLUTTER_ACTOR (p->data), &child_box, flags);

  /* Allocate status bar size */
  clutter_actor_get_preferred_width (CLUTTER_ACTOR (priv->status_bar),
                                     -1,
                                     NULL,
                                     &child_box.x2);
  if (child_box.x2 > box->x2 - box->x1)
    child_box.x2 = box->x2 - box->x1;

  height = box->y2 - box->y1;
  clutter_actor_get_preferred_height (CLUTTER_ACTOR (priv->status_bar),
                                      child_box.x2,
                                      &child_box.y1,
                                      &child_box.y2);
  if (child_box.y2 < height)
    {
      child_box.y1 = height - child_box.y2;
      child_box.y2 = height;
    }
  else
    {
      child_box.y1 = height - child_box.y1;
      child_box.y2 = height;
    }

  clutter_actor_allocate (CLUTTER_ACTOR (priv->status_bar), &child_box, flags);

  CLUTTER_ACTOR_CLASS (mwb_browser_parent_class)->allocate (actor, box, flags);
}

static void
mwb_browser_toggle_fullscreen (MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  GdkWindow *window;

  if (!priv->window)
    return;

  window = gtk_widget_get_window (GTK_WIDGET (priv->window));
  if (gdk_window_get_state (window) & GDK_WINDOW_STATE_FULLSCREEN)
    gtk_window_unfullscreen (GTK_WINDOW (priv->window));
  else
    gtk_window_fullscreen (GTK_WINDOW (priv->window));
}

static gboolean
mwb_browser_bar_show_cb (MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  mwb_radical_bar_focus (mwb_toolbar_get_radical_bar (
                         MWB_TOOLBAR (priv->toolbar)));
  priv->bar_hide_timeout = 0;

  return FALSE;
}

static gboolean
mwb_browser_captured_event (ClutterActor *actor,
                            ClutterEvent *event)
{
  MwbBrowser *browser = MWB_BROWSER (actor);
  MwbBrowserPrivate *priv = browser->priv;

  if (event->type == CLUTTER_KEY_PRESS)
    {
      ClutterKeyEvent *key_event = (ClutterKeyEvent *)event;
      gint n_tabs, tab_id;
      MwbTab *tab;
      MwbTabPicker *tab_picker;

      guint state;

      state = key_event->modifier_state & MWB_UTILS_MODIFIERS_MASK;

      switch (key_event->keyval)
        {
        case CLUTTER_l :
          if (state == CLUTTER_CONTROL_MASK)
            {
              mwb_radical_bar_focus (mwb_toolbar_get_radical_bar (
                                     MWB_TOOLBAR (priv->toolbar)));
              return TRUE;
            }
          break;

        case CLUTTER_d :
          if (state == CLUTTER_MOD1_MASK)
            {
              mwb_radical_bar_focus (mwb_toolbar_get_radical_bar (
                                     MWB_TOOLBAR (priv->toolbar)));
              return TRUE;
            }
          break;

        case CLUTTER_t :
          if (state == CLUTTER_CONTROL_MASK)
            {
              mwb_browser_open (browser, NULL,
                                MWB_BROWSER_OPEN_NEW_TAB |
                                MWB_BROWSER_OPEN_ACTIVATE);
              return TRUE;
            }
          break;

        case CLUTTER_w :
          if (state == CLUTTER_CONTROL_MASK)
            {
              tab_picker = MWB_TAB_PICKER (priv->tab_picker);
              tab_id = mwb_tab_picker_get_current_tab (tab_picker);
              tab = mwb_tab_picker_get_tab (tab_picker, tab_id);

              if (mwb_tab_get_can_close (tab))
                g_signal_emit_by_name (tab, "closed");
            }
          break;

        case CLUTTER_r :
          if (state == CLUTTER_CONTROL_MASK)
            {
              g_signal_emit_by_name (mwb_toolbar_get_radical_bar (
                                       MWB_TOOLBAR (priv->toolbar)),
                                     "reload");
              return TRUE;
            }
          break;

        case CLUTTER_F5 :
          g_signal_emit_by_name (mwb_toolbar_get_radical_bar (
                                   MWB_TOOLBAR (priv->toolbar)),
                                 "reload");
          return TRUE;

        case CLUTTER_Tab :
        case CLUTTER_ISO_Left_Tab :
          /* remember n_tabs includes the + tab */
          tab_picker = MWB_TAB_PICKER (priv->tab_picker);
          n_tabs = mwb_tab_picker_get_n_tabs (tab_picker);
          if (n_tabs > 2)
            {
              tab_id = mwb_tab_picker_get_current_tab (tab_picker);

              if (state == CLUTTER_CONTROL_MASK)
                {
                  tab_id = (tab_id + 1) % (n_tabs - 1);
                  mwb_tab_picker_set_current_tab (tab_picker, tab_id);
                }
              else if (state == (CLUTTER_CONTROL_MASK | CLUTTER_SHIFT_MASK))
                {
                  if (tab_id)
                    tab_id--;
                  else
                    tab_id = n_tabs - 2;
                  mwb_tab_picker_set_current_tab (tab_picker, tab_id);
                }
            }
          break;

        case CLUTTER_F11 :
          mwb_browser_toggle_fullscreen (browser);
          break;

        case CLUTTER_Return :
          if (state == CLUTTER_MOD1_MASK)
            mwb_browser_toggle_fullscreen (browser);
          break;
        }
    }
  else if (event->type == CLUTTER_MOTION)
    {
      gfloat x, y;
      ClutterMotionEvent *mevent = (ClutterMotionEvent *)event;

      if (priv->fullscreen &&
          clutter_actor_transform_stage_point (actor, mevent->x, mevent->y,
                                               &x, &y))
        {
          if (priv->bar_hide_timeout)
            {
              if (y > 3)
                {
                  g_source_remove (priv->bar_hide_timeout);
                  priv->bar_hide_timeout = 0;
                }
            }
          else if ((!priv->bar_visible) && (y < 1))
            {
              priv->bar_hide_timeout =
                g_timeout_add (500,
                               (GSourceFunc)mwb_browser_bar_show_cb,
                               browser);
            }
        }
    }

  if (CLUTTER_ACTOR_CLASS (mwb_browser_parent_class)->captured_event)
    return CLUTTER_ACTOR_CLASS (mwb_browser_parent_class)->
             captured_event (actor, event);
  else
    return FALSE;
}

static void
mwb_browser_map (ClutterActor *actor)
{
  GList *t, *tabs;
  MwbBrowserPrivate *priv = MWB_BROWSER (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_browser_parent_class)->map (actor);

  clutter_actor_map (CLUTTER_ACTOR (priv->toolbar));
  clutter_actor_map (CLUTTER_ACTOR (priv->tab_picker));
  clutter_actor_map (CLUTTER_ACTOR (priv->status_bar));

  tabs = mwb_tab_picker_get_tabs (MWB_TAB_PICKER (priv->tab_picker));
  for (t = tabs; t; t = t->next)
    {
      MwbTab *tab = t->data;
      MwbBrowserTabData *data =
        g_object_get_data (G_OBJECT (tab), "browser-tab-data");
      if (data)
        clutter_actor_map (data->page);
    }
  g_list_free (tabs);
}

static void
mwb_browser_unmap (ClutterActor *actor)
{
  GList *t, *tabs;
  MwbBrowserPrivate *priv = MWB_BROWSER (actor)->priv;

  CLUTTER_ACTOR_CLASS (mwb_browser_parent_class)->unmap (actor);

  clutter_actor_unmap (CLUTTER_ACTOR (priv->toolbar));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->tab_picker));
  clutter_actor_unmap (CLUTTER_ACTOR (priv->status_bar));

  tabs = mwb_tab_picker_get_tabs (MWB_TAB_PICKER (priv->tab_picker));
  for (t = tabs; t; t = t->next)
    {
      MwbTab *tab = t->data;
      MwbBrowserTabData *data =
        g_object_get_data (G_OBJECT (tab), "browser-tab-data");
      if (data)
        clutter_actor_unmap (data->page);
    }
  g_list_free (tabs);
}

static void
mwb_browser_class_init (MwbBrowserClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  ClutterActorClass *actor_class = CLUTTER_ACTOR_CLASS (klass);

  g_type_class_add_private (klass, sizeof (MwbBrowserPrivate));

  object_class->get_property = mwb_browser_get_property;
  object_class->set_property = mwb_browser_set_property;
  object_class->dispose = mwb_browser_dispose;
  object_class->finalize = mwb_browser_finalize;

  actor_class->paint = mwb_browser_paint;
  actor_class->pick = mwb_browser_pick;
  actor_class->allocate = mwb_browser_allocate;
  actor_class->captured_event = mwb_browser_captured_event;
  actor_class->map = mwb_browser_map;
  actor_class->unmap = mwb_browser_unmap;

  g_object_class_install_property (object_class,
                                   PROP_HOME,
                                   g_param_spec_string ("home",
                                                        "Home",
                                                        "Home-page URL.",
                                                        DEFAULT_HOME,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_PRIVATE_HOME,
                                   g_param_spec_string ("private-home",
                                                        "Private home",
                                                        "Private home-page "
                                                        "URL.",
                                                        DEFAULT_PRIVATE_HOME,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));

  g_object_class_install_property (object_class,
                                   PROP_WINDOW,
                                   g_param_spec_object ("window",
                                                        "Window",
                                                        "Parent window.",
                                                        MWB_TYPE_WINDOW,
                                                        G_PARAM_READWRITE |
                                                        G_PARAM_STATIC_NAME |
                                                        G_PARAM_STATIC_NICK |
                                                        G_PARAM_STATIC_BLURB));
}

static void
mwb_browser_update_recent_url (MwbBrowser  *browser,
                               const gchar *url,
                               const gchar *title,
                               gboolean     if_exists)
{
  GtkRecentData data;
  GtkRecentManager *manager;

  if (!url)
    return;

  manager = gtk_recent_manager_get_default ();

  if (!gtk_recent_manager_has_item (manager, url))
    {
      if (if_exists)
        return;
    }

  /* Add to recently used files (user has explicitly typed this URL) */

  /* We don't have the title yet, we'll update this when we get it */
  data.display_name = (gchar *)title;
  data.description = NULL;

  /* FIXME: Probably shouldn't assume this? */
  data.mime_type = "text/html";
  
  /* FIXME: Get these in a reliable way */
  data.app_name = "moblin-web-browser";
  data.app_exec = "moblin-web-browser \"%u\"";
  
  /* TODO: Do we need to add any groups? */
  data.groups = NULL;
  
  data.is_private = FALSE;
  
  gtk_recent_manager_add_full (manager, url, &data);
}

static void
mwb_browser_create_thumbnail_path (const char *thumbnail_size)
{
  gchar *thumb_path = g_build_filename (g_get_home_dir (),
                                        ".thumbnails",
                                        thumbnail_size,
                                        NULL);
  /* Make sure thumbs directory exists */
  g_mkdir_with_parents (thumb_path, 0755);
  g_free (thumb_path);
}

static void
mwb_browser_thumbnail_recent_url (MwbBrowser *browser,
                                  ClutterMozEmbed *mozembed)
{
  gint size;
  guchar *data;
  gint width, height;
  gint dwidth, dheight;
  CoglHandle texture_handle;
  gchar *md5, *thumb, *thumb_path;
  GdkPixbuf *pixbuf, *pixbuf_scaled;

  const gchar *url = clutter_mozembed_get_location (mozembed);

  if (!url)
    return;

  /* Get the texture */
  texture_handle =
    clutter_texture_get_cogl_texture (CLUTTER_TEXTURE (mozembed));

  /* Get the size required */
  size = cogl_texture_get_data (texture_handle,
                                COGL_PIXEL_FORMAT_RGB_888, 0, NULL);

  /* If size is zero, the conversion isn't handled */
  if (!size)
    {
      g_warning ("Unable to convert texture for thumbnail");
      return;
    }

  data = g_malloc (size);
  size = cogl_texture_get_data (texture_handle, COGL_PIXEL_FORMAT_RGB_888, 0,
                                data);

  clutter_texture_get_base_size (CLUTTER_TEXTURE (mozembed), &width, &height);

  pixbuf = gdk_pixbuf_new_from_data (data, GDK_COLORSPACE_RGB, FALSE, 8,
                                     width, height, width * 3, NULL, NULL);

  /* Write thumbnails */
  md5 = g_compute_checksum_for_string (G_CHECKSUM_MD5, url, -1);
  thumb = g_strconcat (md5, ".png", NULL);

  /* Calculate thumb size */
  if (width < height)
    {
      dheight = 256;
      dwidth = width / (gdouble)height * 256.0;
    }
  else
    {
      dwidth = 256;
      dheight = height / (gdouble)width * 256.0;
    }

  mwb_browser_create_thumbnail_path ("normal");
  mwb_browser_create_thumbnail_path ("large");

  /* Normal size thumb */
  thumb_path = g_build_filename (g_get_home_dir (),
                                 ".thumbnails",
                                 "normal",
                                 thumb,
                                 NULL);
  if (!g_file_test (thumb_path, G_FILE_TEST_EXISTS))
    {
      pixbuf_scaled = gdk_pixbuf_scale_simple (pixbuf, dwidth / 2, dheight / 2,
                                               GDK_INTERP_BILINEAR);
      gdk_pixbuf_save (pixbuf_scaled, thumb_path, "png", NULL, NULL);
      g_object_unref (pixbuf_scaled);
    }
  g_free (thumb_path);

  /* Large thumb */
  thumb_path = g_build_filename (g_get_home_dir (),
                                 ".thumbnails",
                                 "large",
                                 thumb,
                                 NULL);
  if (!g_file_test (thumb_path, G_FILE_TEST_EXISTS))
    {
      pixbuf_scaled = gdk_pixbuf_scale_simple (pixbuf, dwidth, dheight,
                                               GDK_INTERP_BILINEAR);
      gdk_pixbuf_save (pixbuf_scaled, thumb_path, "png", NULL, NULL);
      g_object_unref (pixbuf_scaled);
    }
  g_free (thumb_path);
  
  /* Free data */
  g_object_unref (pixbuf);
  g_free (data);
  g_free (thumb);
  g_free (md5);
}

static gboolean
save_session_cb (gpointer data)
{
  mwb_save_session (data);
  return FALSE;
}

static void
queue_save_session (MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  if (priv->save_session_source)
    g_source_remove (priv->save_session_source);
  priv->save_session_source = g_idle_add (save_session_cb, browser);
}

static MwbTab *
mwb_browser_find_tab_for_mozembed (MwbBrowser      *browser,
                                   ClutterMozEmbed *mozembed)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbTabPicker *picker = MWB_TAB_PICKER (priv->tab_picker);
  MwbTab *tab = NULL;
  GList *tabs, *t;

  tabs = mwb_tab_picker_get_tabs (picker);
  for (t = tabs; t; t = t->next)
    {
      MwbBrowserTabData *tab_data
        = g_object_get_data (G_OBJECT (t->data), "browser-tab-data");

      if (!MWB_IS_PAGE (tab_data->page))
        continue;

      if (mwb_page_get_mozembed (MWB_PAGE (tab_data->page)) == mozembed)
        {
          tab = t->data;
          break;
        }
    }
  g_list_free (tabs);

  return tab;
}

static void
mwb_browser_set_is_pinned_for_uri (MwbBrowser *browser,
                                   const gchar *uri,
                                   gboolean is_pinned)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbTabPicker *picker = MWB_TAB_PICKER (priv->tab_picker);
  GList *tabs, *t;

  tabs = mwb_tab_picker_get_tabs (picker);
  for (t = tabs; t; t = t->next)
    {
      MwbBrowserTabData *tab_data
        = g_object_get_data (G_OBJECT (t->data), "browser-tab-data");

      if (tab_data && MWB_IS_PAGE (tab_data->page))
        {
          ClutterMozEmbed *mozembed =
            mwb_page_get_mozembed (MWB_PAGE (tab_data->page));
          const gchar *location = clutter_mozembed_get_location (mozembed);

          if (location && !strcmp (uri, location))
            {
              tab_data->is_pinned = is_pinned;
              if (tab_data->page == priv->current_page)
                {
                  MwbToolbar *toolbar = MWB_TOOLBAR (priv->toolbar);
                  MwbRadicalBar *radical_bar
                    = mwb_toolbar_get_radical_bar (toolbar);
                  mwb_radical_bar_set_pinned (radical_bar, is_pinned);
                }
            }
        }
    }
  g_list_free (tabs);
}

static void
mwb_browser_pinned_page_cb (MhsHistory *history,
                            const gchar *title,
                            const gchar *uri,
                            gint visit_time,
                            MwbBrowser *browser)
{
  mwb_browser_set_is_pinned_for_uri (browser, uri, TRUE);
}

static void
mwb_browser_unpinned_page_cb (MhsHistory *history,
                              const gchar *uri,
                              MwbBrowser *browser)
{
  mwb_browser_set_is_pinned_for_uri (browser, uri, FALSE);
}

static void
mwb_browser_get_is_pinned_cb (MhsHistory *history,
                              gboolean is_pinned,
                              const GError *error,
                              gpointer user_data)
{
  MwbBrowserTabData *tab_data = user_data;

  if (error)
    g_warning ("pinning error: %s\n", error->message);
  else
    {
      MwbBrowser *self = MWB_BROWSER (tab_data->browser);
      MwbBrowserPrivate *priv = self->priv;

      tab_data->is_pinned = is_pinned;

      if (priv->current_page == tab_data->page)
        {
          MwbRadicalBar *radical_bar
            = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
          mwb_radical_bar_set_pinned (radical_bar, is_pinned);
        }
    }

  tab_data->is_pinned_handler = 0;
}

static void
mwb_browser_request_pinned_status (MwbBrowser *browser,
                                   MwbBrowserTabData *tab_data)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->history && tab_data && MWB_IS_PAGE (tab_data->page))
    {
      ClutterMozEmbed *mozembed =
        mwb_page_get_mozembed (MWB_PAGE (tab_data->page));
      const gchar *location = clutter_mozembed_get_location (mozembed);

      if (tab_data->is_pinned_handler)
        {
          mhs_history_cancel (priv->history, tab_data->is_pinned_handler);
          tab_data->is_pinned_handler = 0;
        }

      if (location)
        tab_data->is_pinned_handler =
          mhs_history_get_is_page_pinned (priv->history,
                                          location,
                                          mwb_browser_get_is_pinned_cb,
                                          tab_data,
                                          NULL);
    }
}

static void
mwb_browser_location_notify_cb (ClutterMozEmbed *mozembed,
                                GParamSpec      *pspec,
                                MwbBrowser      *browser)
{
  const gchar *location;
  gchar *request;

  MwbBrowserPrivate *priv = browser->priv;
  MwbRadicalBar *bar =
    mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
  MwbTab *tab;
  gboolean is_current_tab;

  location = clutter_mozembed_get_location (mozembed);
  request = g_object_get_data (G_OBJECT (mozembed), "request");

  /* pspec is only set if this is a real update from the back end */
  if (pspec)
    {
      /* Prevent display of about:blank URL as mozembed comes online */
      if (location && !strcmp (location, "about:blank"))
        return;

      /* Real update came from the back end, remove user request string */
      if (request)
        {
          g_object_set_data (G_OBJECT (mozembed), "request", NULL);
          g_free (request);
          request = NULL;
        }
    }

  if (request)
    location = request;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);
  if (tab)
    {
      MwbBrowserTabData *tab_data
        = g_object_get_data (G_OBJECT (tab), "browser-tab-data");
      if (tab_data)
        mwb_browser_request_pinned_status (browser, tab_data);
    }

  is_current_tab =
    priv->current_page && MWB_IS_PAGE (priv->current_page) &&
    (mozembed == mwb_page_get_mozembed (MWB_PAGE (priv->current_page)));

  if (is_current_tab && priv->user_initiated_url && location &&
      (!g_str_equal (location, DEFAULT_HOME)) &&
      /* Don't add private tabs to the recent manager */
      (tab == NULL || !mwb_tab_get_private (tab)))
    {
      mwb_browser_update_recent_url (browser,
                                     location,
                                     NULL,
                                     FALSE);
      priv->user_initiated_url = FALSE;
    }

  if (is_current_tab)
    {
      mwb_radical_bar_set_text (bar, location);

      /* If we're on the home page, set the next action to be user-initiated */
      if (priv->home && location && g_str_equal (location, priv->home))
        priv->user_initiated_url = TRUE;
    }

  queue_save_session (browser);
}

static void
mwb_browser_update_title (MwbBrowser *browser,
                         const gchar *title)
{
  gchar *full_title;
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->window)
    return;

  full_title = g_strconcat (_("Moblin web browser - "), title, NULL);
  gtk_window_set_title (GTK_WINDOW (priv->window), full_title);
  g_free (full_title);
}

static void
mwb_browser_title_notify_cb (ClutterMozEmbed *mozembed,
                             GParamSpec      *pspec,
                             MwbBrowser      *browser)
{
  const gchar *title;

  MwbTab *tab = NULL;
  MwbBrowserPrivate *priv = browser->priv;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);

  if (!tab)
    return;

  title = clutter_mozembed_get_title (mozembed);

  if (!title || title[0] == '\0')
    title = _("Untitled");

  mwb_tab_set_text (tab, title);
  if (priv->current_page && MWB_IS_PAGE (priv->current_page) &&
      (mwb_page_get_mozembed (MWB_PAGE (priv->current_page)) == mozembed))
    mwb_browser_update_title (browser, title);
}

static void
mwb_browser_icon_notify_cb (ClutterMozEmbed *mozembed,
                            GParamSpec      *pspec,
                            MwbBrowser      *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  const gchar *location, *icon;
  MwbTab *tab;
  MwbBrowserTabData *tab_data;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);
  if (tab == NULL)
    return;
  tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");
  if (tab_data == NULL)
    return;

  icon = clutter_mozembed_get_icon (mozembed);

  if (icon)
    {
      location = clutter_mozembed_get_location (mozembed);

      if (location)
        {
          GError *error = NULL;

          mhs_history_set_favicon_url (priv->history, location, icon,
                                       &error);
          tab_data->got_icon = TRUE;

          if (error)
            {
              g_warning ("%s", error->message);
              g_error_free (error);
            }
          else
            mwb_browser_start_loading_favicon (browser, tab_data);
        }
    }
}

static void
mwb_browser_tab_icon_notify_cb (MwbTab     *tab,
                                GParamSpec *pspec,
                                MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbBrowserTabData *tab_data = g_object_get_data (G_OBJECT (tab),
                                                   "browser-tab-data");

  if (tab_data->page == priv->current_page)
    {
      MwbRadicalBar *radical_bar;
      ClutterActor *icon;

      radical_bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
      icon = mwb_tab_get_icon (tab);
      mwb_radical_bar_set_icon (radical_bar,
                                icon ? clutter_clone_new (icon) : NULL);
    }
}

static void
mwb_browser_net_start_cb (ClutterMozEmbed *mozembed, MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbRadicalBar *bar =
    mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
  MwbTab *tab;
  MwbBrowserTabData *tab_data;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);
  if (tab)
    {
      tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");
      if (tab_data)
        {
          mwb_browser_stop_loading_favicon (browser, tab_data);
          tab_data->got_icon = FALSE;
          mwb_tab_set_icon (tab, NULL);
        }
    }

  if (priv->current_page && MWB_IS_PAGE (priv->current_page) &&
      (mozembed == mwb_page_get_mozembed (MWB_PAGE (priv->current_page))))
    mwb_radical_bar_set_loading (bar, TRUE);
}

static void
mwb_browser_net_stop_cb (ClutterMozEmbed *mozembed, MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbRadicalBar *bar =
    mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
  MwbTab *tab;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);
  if (tab)
    {
      MwbBrowserTabData *tab_data;

      tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");

      if (tab_data && !tab_data->got_icon)
        {
          const gchar *location;

          /* If the page didn't contain an icon link then attempt to use
             the default /favicon.ico url */
          location = clutter_mozembed_get_location (mozembed);

          if (location)
            {
              GError *error = NULL;

              mhs_history_set_default_favicon_url (priv->history, location,
                                                   &error);

              if (error)
                {
                  g_warning ("%s", error->message);
                  g_error_free (error);
                }

              mwb_browser_start_loading_favicon (browser, tab_data);
            }
        }
    }

  if (priv->current_page && MWB_IS_PAGE (priv->current_page) &&
      (mozembed == mwb_page_get_mozembed (MWB_PAGE (priv->current_page))))
    mwb_radical_bar_set_loading (bar, FALSE);

  /* Update thumbnail on recent info, and add title to force
   * GtkRecentManager to issue an update signal. We don't want to do
   * this for private tabs
   */
  if (tab == NULL || !mwb_tab_get_private (tab))
    {
      mwb_browser_thumbnail_recent_url (browser, mozembed);
      mwb_browser_update_recent_url (browser,
                                     clutter_mozembed_get_location (mozembed),
                                     clutter_mozembed_get_title (mozembed),
                                     TRUE);
    }
}

static void
mwb_browser_pin_button_clicked_cb (MwbRadicalBar *radical_bar,
                                   MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbTabPicker *tab_picker = MWB_TAB_PICKER (priv->tab_picker);
  MwbTab *tab;
  MwbBrowserTabData *tab_data;

  tab = mwb_tab_picker_get_tab (tab_picker, priv->current_tab);
  tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");

  if (tab_data && MWB_IS_PAGE (tab_data->page))
    {
      ClutterMozEmbed *mozembed =
        mwb_page_get_mozembed (MWB_PAGE (tab_data->page));
      const gchar *location = clutter_mozembed_get_location (mozembed);

      if (location && priv->history)
        {
          tab_data->is_pinned = !tab_data->is_pinned;

          if (tab_data->is_pinned)
            mhs_history_pin_page (priv->history,
                                  location,
                                  mwb_tab_get_text (tab));
          else
            mhs_history_unpin_page (priv->history, location);

          /* Update the pin icon */
          if (priv->current_page == tab_data->page)
            {
              MwbRadicalBar *radical_bar =
                mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
              mwb_radical_bar_set_pinned (radical_bar, tab_data->is_pinned);
            }

          /* Rerequest the pinned status so that the pin icon will
             revert if it didn't work */
          mwb_browser_request_pinned_status (browser, tab_data);
        }
    }
}

static void
mwb_browser_settings_cb (ClutterMozEmbed *mozembed, MwbBrowser *browser)
{
  mwb_browser_open (browser,
                    "chrome://mwbpages/content/settings.xhtml",
                    MWB_BROWSER_OPEN_USE_EXISTING |
                    MWB_BROWSER_OPEN_NEW_TAB |
                    MWB_BROWSER_OPEN_ACTIVATE);
}

static void
mwb_browser_progress_cb (ClutterMozEmbed *mozembed,
                         gdouble          progress,
                         MwbBrowser      *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  MwbRadicalBar *bar;

  if (!priv->current_page || !MWB_IS_PAGE (priv->current_page) ||
      mwb_page_get_mozembed (MWB_PAGE (priv->current_page)) != mozembed)
    return;

  bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
  mwb_radical_bar_set_progress (bar, progress);
}

static void
mwb_browser_new_window_cb (ClutterMozEmbed  *mozembed,
                           ClutterMozEmbed **new_mozembed,
                           guint             chromemask,
                           MwbBrowser       *browser)
{
  MwbTab *tab;
  ClutterActor *page;
  MwbBrowserPrivate *priv = browser->priv;

  if (chromemask & MOZ_HEADLESS_FLAG_OPENASCHROME)
    return;

  *new_mozembed = CLUTTER_MOZEMBED (mwb_mozembed_new_for_new_window ());

  page =
    CLUTTER_ACTOR (mwb_page_new_with_mozembed (MWB_MOZEMBED (*new_mozembed)));
  mwb_page_set_download_manager (MWB_PAGE (page),
                                 MWB_DOWNLOAD_MANAGER (priv->dlman));

  tab = mwb_browser_add_tab (browser, NULL, page, TRUE, TRUE);
  if (clutter_mozembed_get_private (mozembed))
    mwb_tab_set_private (tab, TRUE);
}

static void
mwb_browser_closed_cb (ClutterMozEmbed *mozembed,
                       MwbBrowser      *browser)
{
  MwbTab *tab = NULL;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);

  if (!tab)
    return;

  mwb_browser_tab_closed_cb (tab, browser);
}

static void
mwb_browser_link_msg_cb (ClutterMozEmbed *mozembed,
                         const gchar     *message,
                         MwbBrowser      *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->current_page && MWB_IS_PAGE (priv->current_page) &&
      (mwb_page_get_mozembed (MWB_PAGE (priv->current_page)) == mozembed))
    mwb_status_bar_set_text (MWB_STATUS_BAR (priv->status_bar), message, TRUE);
}

static void
mwb_browser_nav_changed_cb (ClutterMozEmbed *mozembed,
                            GParamSpec      *pspec,
                            MwbBrowser      *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->current_page && MWB_IS_PAGE (priv->current_page) &&
      (mwb_page_get_mozembed (MWB_PAGE (priv->current_page)) == mozembed))
    {
      mwb_toolbar_set_can_go_back (MWB_TOOLBAR (priv->toolbar),
                                   clutter_mozembed_can_go_back (mozembed));
      mwb_toolbar_set_can_go_forward (MWB_TOOLBAR (priv->toolbar),
                                      clutter_mozembed_can_go_forward (
                                      mozembed));
    }
}

static GdkCursor *
get_gdk_cursor (MozHeadlessCursorType type, MozHeadlessCursor *special)
{
  GdkCursor *gdkcursor;
  /* gboolean is_special = FALSE; */

  gdkcursor = cursor_cache[type];
  if (gdkcursor)
    return gdkcursor;

  switch (type)
    {
    case MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR:
      gdkcursor = gdk_cursor_new(GDK_LEFT_PTR);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_WATCH:
      gdkcursor = gdk_cursor_new(GDK_WATCH);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_XTERM:
      gdkcursor = gdk_cursor_new(GDK_XTERM);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_HAND2:
      gdkcursor = gdk_cursor_new(GDK_HAND2);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_TOP_SIDE:
      gdkcursor = gdk_cursor_new(GDK_TOP_SIDE);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_BOTTOM_SIDE:
      gdkcursor = gdk_cursor_new(GDK_BOTTOM_SIDE);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_LEFT_SIDE:
      gdkcursor = gdk_cursor_new(GDK_LEFT_SIDE);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_RIGHT_SIDE:
      gdkcursor = gdk_cursor_new(GDK_RIGHT_SIDE);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_TOP_LEFT_CORNER:
      gdkcursor = gdk_cursor_new(GDK_TOP_LEFT_CORNER);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_BOTTOM_RIGHT_CORNER:
      gdkcursor = gdk_cursor_new(GDK_BOTTOM_RIGHT_CORNER);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_TOP_RIGHT_CORNER:
      gdkcursor = gdk_cursor_new(GDK_TOP_RIGHT_CORNER);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_BOTTOM_LEFT_CORNER:
      gdkcursor = gdk_cursor_new(GDK_BOTTOM_LEFT_CORNER);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_CROSSHAIR:
      gdkcursor = gdk_cursor_new(GDK_CROSSHAIR);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_FLEUR:
      gdkcursor = gdk_cursor_new(GDK_FLEUR);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_QUESTION_ARROW:
      gdkcursor = gdk_cursor_new(GDK_QUESTION_ARROW);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_SB_V_DOUBLE_ARROW:
      gdkcursor = gdk_cursor_new(GDK_SB_V_DOUBLE_ARROW);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_SB_H_DOUBLE_ARROW:
      gdkcursor = gdk_cursor_new(GDK_SB_H_DOUBLE_ARROW);
      break;
    case MOZ_HEADLESS_CURSOR_TYPE_PLUS:
      gdkcursor = gdk_cursor_new(GDK_PLUS);
      break;

    case MOZ_HEADLESS_CURSOR_TYPE_HAND_GRAB:
    case MOZ_HEADLESS_CURSOR_TYPE_HAND_GRABBING:
    case MOZ_HEADLESS_CURSOR_TYPE_COPY:
    case MOZ_HEADLESS_CURSOR_TYPE_ALIAS:
    case MOZ_HEADLESS_CURSOR_TYPE_CONTEXT_MENU:
    case MOZ_HEADLESS_CURSOR_TYPE_SPINNING:
    case MOZ_HEADLESS_CURSOR_TYPE_ZOOM_IN:
    case MOZ_HEADLESS_CURSOR_TYPE_ZOOM_OUT:
    case MOZ_HEADLESS_CURSOR_TYPE_NOT_ALLOWED:
    case MOZ_HEADLESS_CURSOR_TYPE_COL_RESIZE:
    case MOZ_HEADLESS_CURSOR_TYPE_ROW_RESIZE:
    case MOZ_HEADLESS_CURSOR_TYPE_VERTICAL_TEXT:
    case MOZ_HEADLESS_CURSOR_TYPE_NESW_RESIZE:
    case MOZ_HEADLESS_CURSOR_TYPE_NWSE_RESIZE:
    case MOZ_HEADLESS_CURSOR_TYPE_NONE:
#if 0
      is_special = TRUE;
      g_assert (special);
#endif

    case MOZ_HEADLESS_CURSOR_TYPE_N_SPECIAL:
    case MOZ_HEADLESS_CURSOR_TYPE_N_TYPES:
      g_warning ("invalid cursor type");
      goto error;
    }

#if 0
  if (is_special)
    {
      GdkColor fg, bg;
      GdkPixmap *bitmap;
      GdkPixmap *mask;

      gdk_color_parse ("#000000", &fg);
      gdk_color_parse ("#ffffff", &bg);

      bitmap =
        gdk_bitmap_create_from_data (NULL, (char *)special->bits, 32, 32);
      if (!bitmap)
        goto error;

      mask = gdk_bitmap_create_from_data (NULL, (char *)special->mask_bits,
                                          32, 32);
      if (!mask) {
          g_object_unref (bitmap);
          goto error;
      }

      gdkcursor = gdk_cursor_new_from_pixmap (bitmap, mask, &fg, &bg,
                                              special->hot_x,
                                              special->hot_y);

      g_object_unref (mask);
      g_object_unref (bitmap);
    }
#endif

  cursor_cache[type] = gdkcursor;

  return gdkcursor;

error:
  gdkcursor = cursor_cache[MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR];
  if (gdkcursor)
    return gdkcursor;

  cursor_cache[MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR] =
    gdk_cursor_new(GDK_LEFT_PTR);
  return cursor_cache[MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR];
}

static void
update_cursor (MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;
  GdkCursor *gdkcursor;
  GdkWindow *gdkwindow;

  if (priv->use_mozembed_cursor &&
      priv->mozembed_cursor != MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR)
    gdkcursor = get_gdk_cursor (priv->mozembed_cursor, NULL);
  else
    {
      MwbRadicalBar *bar =
        mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));

      if (mwb_radical_bar_get_loading (bar))
        gdkcursor = get_gdk_cursor (MOZ_HEADLESS_CURSOR_TYPE_WATCH, NULL);
      else
        gdkcursor = NULL;
    }

  gdkwindow = gtk_widget_get_window (GTK_WIDGET (priv->window));
  gdk_window_set_cursor (gdkwindow, gdkcursor);
}

static void
set_use_mozembed_cursor (MwbBrowser *browser, gboolean use_mozembed_cursor)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->use_mozembed_cursor != use_mozembed_cursor)
    {
      priv->use_mozembed_cursor = use_mozembed_cursor;
      update_cursor (browser);
    }
}

static void
mwb_browser_cursor_changed_cb (ClutterMozEmbed *mozembed,
                               GParamSpec      *pspec,
                               MwbBrowser      *browser)
{
  MwbTab *tab = NULL;
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->window)
    return;

  tab = mwb_browser_find_tab_for_mozembed (browser, mozembed);
  if (!tab)
    return;

  priv->mozembed_cursor = clutter_mozembed_get_cursor (mozembed);

  if (priv->use_mozembed_cursor)
    update_cursor (browser);
}

static void
mwb_browser_security_changed_cb (ClutterMozEmbed *mozembed,
                                 GParamSpec      *pspec,
                                 MwbBrowser      *browser)
{
  guint security;
  MwbBrowserPrivate *priv = browser->priv;
  MwbRadicalBar *bar =
    mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));

  security = clutter_mozembed_get_security (mozembed);

  if (priv->current_page &&
      (mozembed == mwb_page_get_mozembed (MWB_PAGE (priv->current_page))))
    mwb_radical_bar_set_security (bar, security);
}

static void
mwb_browser_page_focus_in_cb (MwbPage    *page,
                              MwbBrowser *self)
{
  MwbBrowserPrivate *priv = self->priv;

  if (priv->fullscreen)
    mwb_browser_toggle_toolbar (self, FALSE);

  priv->bar_visible = FALSE;
}

static void
mwb_browser_page_focus_out_cb (MwbPage    *page,
                               MwbBrowser *self)
{
  MwbBrowserPrivate *priv = self->priv;

  if (priv->fullscreen)
    mwb_browser_toggle_toolbar (self, TRUE);
  else
    priv->bar_visible = TRUE;
}

static void
mwb_browser_new_tab_cb (MwbMozEmbed *mozembed,
                        const gchar *url,
                        MwbBrowser  *self)
{
  MwbBrowserOpenFlags flags = (MWB_BROWSER_OPEN_NEW_TAB |
                               MWB_BROWSER_OPEN_ACTIVATE);

  if (clutter_mozembed_get_private (CLUTTER_MOZEMBED (mozembed)))
    flags |= MWB_BROWSER_OPEN_PRIVATE;

  mwb_browser_open (self, url, flags);
}

static void
mwb_browser_embed_connect (MwbBrowser *self, MwbPage *page)
{
  ClutterMozEmbed *embed = mwb_page_get_mozembed (page);

  g_signal_connect (page, "key-focus-in",
                    G_CALLBACK (mwb_browser_page_focus_in_cb), self);
  g_signal_connect (page, "key-focus-out",
                    G_CALLBACK (mwb_browser_page_focus_out_cb), self);

  if (!embed)
    return;

  g_signal_connect (embed, "notify::location",
                    G_CALLBACK (mwb_browser_location_notify_cb), self);
  g_signal_connect (embed, "notify::icon",
                    G_CALLBACK (mwb_browser_icon_notify_cb), self);
  g_signal_connect (embed, "notify::title",
                    G_CALLBACK (mwb_browser_title_notify_cb), self);
  g_signal_connect (embed, "net-start",
                    G_CALLBACK (mwb_browser_net_start_cb), self);
  g_signal_connect (embed, "net-stop",
                    G_CALLBACK (mwb_browser_net_stop_cb), self);
  g_signal_connect (embed, "progress",
                    G_CALLBACK (mwb_browser_progress_cb), self);
  g_signal_connect (embed, "new-window",
                    G_CALLBACK (mwb_browser_new_window_cb), self);
  g_signal_connect (embed, "closed",
                    G_CALLBACK (mwb_browser_closed_cb), self);
  g_signal_connect (embed, "link-message",
                    G_CALLBACK (mwb_browser_link_msg_cb), self);
  g_signal_connect (embed, "notify::can-go-back",
                    G_CALLBACK (mwb_browser_nav_changed_cb), self);
  g_signal_connect (embed, "notify::can-go-forward",
                    G_CALLBACK (mwb_browser_nav_changed_cb), self);
  g_signal_connect (embed, "notify::cursor",
                    G_CALLBACK (mwb_browser_cursor_changed_cb), self);
  g_signal_connect (embed, "notify::security",
                    G_CALLBACK (mwb_browser_security_changed_cb), self);
  g_signal_connect (embed, "new-tab",
                    G_CALLBACK (mwb_browser_new_tab_cb), self);

  mwb_browser_location_notify_cb (embed, NULL, self);
  mwb_browser_title_notify_cb (embed, NULL, self);
  mwb_browser_nav_changed_cb (embed, NULL, self);
  mwb_browser_security_changed_cb (embed, NULL, self);
}

static void
mwb_browser_embed_disconnect (MwbBrowser *self, MwbPage *page)
{
  ClutterMozEmbed *embed = mwb_page_get_mozembed (page);

  g_signal_handlers_disconnect_by_func (page,
                                        mwb_browser_page_focus_in_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (page,
                                        mwb_browser_page_focus_out_cb,
                                        self);

  if (!embed)
    return;

  mwb_browser_net_stop_cb (embed, self);

  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_location_notify_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_title_notify_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_icon_notify_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_net_start_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_net_stop_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_progress_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_new_window_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_closed_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_link_msg_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_nav_changed_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_cursor_changed_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_security_changed_cb,
                                        self);
  g_signal_handlers_disconnect_by_func (embed,
                                        mwb_browser_new_tab_cb,
                                        self);
}

static void
mwb_browser_hide_complete_cb (ClutterAnimation *animation,
                              ClutterActor     *actor)
{
  clutter_actor_hide (actor);
}

static gboolean
mwb_browser_page_enter_cb (ClutterActor *actor,
                           ClutterCrossingEvent *event,
                           MwbBrowser *self)
{
  set_use_mozembed_cursor (self, TRUE);

  return FALSE;
}

static gboolean
mwb_browser_page_leave_cb (ClutterActor *actor,
                           ClutterCrossingEvent *event,
                           MwbBrowser *self)
{
  set_use_mozembed_cursor (self, FALSE);

  return FALSE;
}

static void
mwb_browser_tab_activated_cb (MwbTabPicker *picker,
                              MwbTab       *tab,
                              MwbBrowser   *self)
{
  ClutterAnimation *animation;
  MwbBrowserTabData *tab_data;

  MwbBrowserPrivate *priv = self->priv;

  mwb_browser_update_title (self, mwb_tab_get_text (tab));
  mwb_tab_picker_set_preview_mode (MWB_TAB_PICKER (priv->tab_picker), FALSE);

  tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");

  if (tab_data)
    {
      MwbRadicalBar *bar;
      ClutterMozEmbed *mozembed;
      ClutterActor *page = CLUTTER_ACTOR (tab_data->page);

      priv->current_tab =
        mwb_tab_picker_get_current_tab (MWB_TAB_PICKER (priv->tab_picker));

      if (priv->current_page == tab_data->page)
        return;

      disconnect_page_enter_leave_handlers (self);
      set_use_mozembed_cursor (self, FALSE);

      /* Animate old page away */
      if (priv->current_page)
        {
          clutter_actor_set_reactive (priv->current_page, FALSE);
          if (!priv->skip_tab_transition)
            {
              animation = clutter_actor_animate (priv->current_page,
                                                 CLUTTER_EASE_IN_SINE,
                                                 250,
                                                 "opacity", 0x00,
                                                 NULL);
              g_signal_connect (animation, "completed",
                                G_CALLBACK (mwb_browser_hide_complete_cb),
                                priv->current_page);
            }
          else
            clutter_actor_hide (priv->current_page);
        }

      /* Animate in new page */
      if (!CLUTTER_ACTOR_IS_VISIBLE (page))
        {
          clutter_actor_set_opacity (page, 0x00);
          clutter_actor_show (page);
        }

      if (!priv->skip_tab_transition)
        {
          ClutterAnimation *animation;

          /* Make sure an old hide animation isn't hanging around */
          animation = clutter_actor_get_animation (page);
          if (animation)
            clutter_animation_completed (animation);

          clutter_actor_show (page);

          clutter_actor_animate (page,
                                 CLUTTER_EASE_IN_SINE,
                                 250,
                                 "opacity", 0xff,
                                 NULL);
        }
      else
        clutter_actor_set_opacity (page, 0xff);

      priv->current_page = (ClutterActor *)tab_data->page;
      priv->skip_tab_transition = FALSE;

      clutter_actor_set_reactive (page, TRUE);
      bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));

      if (MWB_IS_PAGE (tab_data->page))
        {
          MwbPage *page = MWB_PAGE (tab_data->page);

          /* Focus */
          clutter_actor_grab_key_focus (tab_data->page);

          /* Make sure the toolbar status is correct */
          mozembed = mwb_page_get_mozembed (page);

          priv->user_initiated_url = FALSE;

          mwb_browser_location_notify_cb (mozembed, NULL, self);
          mwb_browser_title_notify_cb (mozembed, NULL, self);
          mwb_browser_nav_changed_cb (mozembed, NULL, self);
          mwb_browser_security_changed_cb (mozembed, NULL, self);

          if (clutter_mozembed_is_loading (mozembed))
            {
              gdouble progress = clutter_mozembed_get_progress (mozembed);
              mwb_radical_bar_set_loading (bar, TRUE);
              mwb_radical_bar_set_progress (bar, progress);
            }
          else
            mwb_radical_bar_set_loading (bar, FALSE);

          priv->page_enter_handler =
            g_signal_connect (priv->current_page, "enter-event",
                              G_CALLBACK (mwb_browser_page_enter_cb), self);
          priv->page_leave_handler =
            g_signal_connect (priv->current_page, "leave-event",
                              G_CALLBACK (mwb_browser_page_leave_cb), self);
        }
      else if (MWB_IS_DOWNLOAD_MANAGER (tab_data->page))
        {
          MwbRadicalBar *bar =
            mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
          mwb_radical_bar_set_text (bar, _("Downloads"));
        }

      mwb_radical_bar_set_pinned (bar, tab_data->is_pinned);

      /* Update with the icon from the new tab */
      mwb_browser_tab_icon_notify_cb (tab, NULL, self);
    }
  else
    {
      MwbRadicalBar *bar =
        mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));

      /* This must be the 'new page' tab, open a new tab */
      mwb_browser_open (self, NULL,
                        MWB_BROWSER_OPEN_NEW_TAB |
                        MWB_BROWSER_OPEN_ACTIVATE);

      /* Focus the location entry */
      mwb_radical_bar_focus (bar);
    }
}

static void
mwb_browser_wrapped_open (ClutterMozEmbed *mozembed, const gchar *url)
{
  gchar *request;

  request = (gchar *)g_object_get_data (G_OBJECT (mozembed), "request");
  g_free (request);

  request = g_strdup (url);
  g_object_set_data (G_OBJECT (mozembed), "request", request);

  clutter_mozembed_open (mozembed, url);
}

static void
mwb_browser_go_cb (MwbRadicalBar *radical,
                   const gchar   *url,
                   MwbBrowser    *browser)
{
  ClutterMozEmbed *mozembed;
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->current_page)
    return;

  /* If we have search enabled in the radical bar and this looks like
   * it isn't a valid URL, search for it.
   * FIXME: We should do this *after* finding it's not valid really, but
   * this should be fine 99.9% of the time. Revisit when someone reports
   * an error.
   * Current logic is to search for URLs entered with
   * no slash, a space and no top-level-domain.
   */
  if (!strstr (url, "/") && strchr (url, ' '))
    {
      gboolean search_enabled;

      MwbAcList *ac_list =
        mwb_radical_bar_get_ac_list (
          mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar)));

      g_object_get (G_OBJECT (ac_list),
                    "search-enabled", &search_enabled,
                    NULL);

      if (search_enabled)
        {
          gboolean has_tld = FALSE;
          GList *t, *tlds = mwb_ac_list_get_tld_suggestions (ac_list);
          for (t = tlds; t; t = t->next)
            {
              const gchar *tld_string = (const gchar *)t->data;
              if (strstr (url, tld_string))
                {
                  has_tld = TRUE;
                  break;
                }
            }
          g_list_free (tlds);

          if (!has_tld)
            {
              /* Perform a search */
              mwb_ac_list_set_search_text (ac_list, url);
              mwb_ac_list_set_selection (ac_list, 0);
              g_signal_emit_by_name (ac_list, "activate");
              return;
            }
        }
    }

  if (!MWB_IS_PAGE (priv->current_page))
    {
      mwb_browser_open (browser,
                        url,
                        MWB_BROWSER_OPEN_USER_INITIATED |
                        MWB_BROWSER_OPEN_ACTIVATE |
                        MWB_BROWSER_OPEN_NEW_TAB);
    }
  else
    {
      mozembed = mwb_page_get_mozembed (MWB_PAGE (priv->current_page));
      mwb_browser_wrapped_open (mozembed, url);
      clutter_actor_grab_key_focus (priv->current_page);
      priv->user_initiated_url = TRUE;
    }
}

static void
mwb_browser_reload_cb (MwbRadicalBar *radical,
                       MwbBrowser    *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->current_page || !MWB_IS_PAGE (priv->current_page))
    return;

  /* TODO: Decide if this should be refresh or reload? */
  clutter_mozembed_refresh (
    mwb_page_get_mozembed (MWB_PAGE (priv->current_page)));
}

static void
mwb_browser_stop_cb (MwbRadicalBar *radical,
                     MwbBrowser     *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->current_page || !MWB_IS_PAGE (priv->current_page))
    return;

  clutter_mozembed_stop (mwb_page_get_mozembed (MWB_PAGE (priv->current_page)));
}

static void
mwb_browser_toolbar_travel_cb (MwbToolbar  *toolbar,
                               gint         direction,
                               MwbBrowser   *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  /* TODO: Handle ABS(direction) != 1 */

  if (!priv->current_page || !MWB_IS_PAGE (priv->current_page))
    return;

  if (direction > 0)
    clutter_mozembed_forward (
      mwb_page_get_mozembed (MWB_PAGE (priv->current_page)));
  else if (direction < 0)
    clutter_mozembed_back (
      mwb_page_get_mozembed (MWB_PAGE (priv->current_page)));
}

static gboolean
mwb_browser_window_delete_cb (GtkWidget  *window,
                              GdkEvent   *event,
                              MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (mwb_download_model_get_active (
        mwb_download_manager_get_model (MWB_DOWNLOAD_MANAGER (priv->dlman))))
    {
      MwbTab *tab = mwb_browser_get_downloads_tab (browser);
      if (tab)
        {
          MwbTabPicker *picker = MWB_TAB_PICKER (priv->tab_picker);
          mwb_tab_picker_set_current_tab (picker,
                                          mwb_tab_picker_get_tab_no (picker,
                                                                     tab));
          mwb_tab_alert (tab);

          return TRUE;
        }
    }

  return FALSE;
}

static void
mwb_browser_toolbar_quit_cb (MwbToolbar *toolbar,
                             MwbBrowser *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (!mwb_browser_window_delete_cb (GTK_WIDGET (priv->window),
                                     NULL,
                                     browser))
    gtk_widget_destroy (GTK_WIDGET (priv->window));
}

static void
tab_picker_preview_mode_notify (MwbTabPicker *picker,
                                GParamSpec   *pspec,
                                MwbBrowser    *self)
{
  GList *tabs, *t;
  
  gboolean preview_mode = mwb_tab_picker_get_preview_mode (picker);

  clutter_actor_set_name (CLUTTER_ACTOR (picker),
                          preview_mode ? "tab-picker-preview" : NULL);

  tabs = mwb_tab_picker_get_tabs (picker);
  for (t = tabs; t; t = t->next)
    {
      MwbTab *tab;
      MwbBrowserTabData *tab_data;

      tab = MWB_TAB (t->data);
      tab_data = g_object_get_data (G_OBJECT (t->data), "browser-tab-data");

      if (!tab_data)
        continue;

      if (preview_mode)
        {
          /* Show all pages so that the preview clones work correctly */
          if (!CLUTTER_ACTOR_IS_VISIBLE (tab_data->page))
            {
              clutter_actor_set_opacity (tab_data->page, 0x00);
              clutter_actor_show (tab_data->page);
            }
        }
      else
        {
          /* Hide invisible pages */
          if (clutter_actor_get_opacity (tab_data->page) == 0x0)
            clutter_actor_hide (tab_data->page);
        }
    }
  g_list_free (tabs);
}

static void
mwb_browser_chooser_clicked_cb (MwbTabPicker *picker, MwbBrowser *self)
{
  gboolean preview_mode = !mwb_tab_picker_get_preview_mode (picker);
  mwb_tab_picker_set_preview_mode (picker, preview_mode);
}

static MwbTab *
mwb_browser_get_downloads_tab (MwbBrowser *self)
{
  MwbTab *tab = NULL;
  MwbBrowserPrivate *priv = self->priv;
  MwbTabPicker *picker = MWB_TAB_PICKER (priv->tab_picker);
  GList *t, *tabs = mwb_tab_picker_get_tabs (picker);

  for (t = tabs; t; t = t->next)
    {
      MwbBrowserTabData *tab_data;

      tab_data = g_object_get_data (G_OBJECT (t->data), "browser-tab-data");

      if (tab_data && (tab_data->page == (ClutterActor *)priv->dlman))
        {
          tab = MWB_TAB (t->data);
          break;
        }
    }
  g_list_free (tabs);

  return tab;
}

static void
mwb_browser_download_added_cb (ClutterActor *dlman, MwbBrowser *self)
{
  MwbTab *tab = NULL;
  MwbBrowserPrivate *priv = self->priv;

  /* Show the download manager */
  if (!clutter_actor_get_parent (dlman))
    {
      tab = mwb_browser_add_tab (self, _("Downloads"), dlman, TRUE, TRUE);
      mwb_tab_set_icon (tab, clutter_texture_new_from_file (
        PKGDATADIR "/downloads-favicon.png", NULL));
    }
  else
    tab = mwb_browser_get_downloads_tab (self);

  if (tab)
    {
      MwbTabPicker *picker = MWB_TAB_PICKER (priv->tab_picker);

      /* Disable closing until downloads have finished */
      mwb_tab_set_can_close (tab, FALSE);

      /* Activate tab */
      mwb_tab_picker_set_current_tab (picker,
                                      mwb_tab_picker_get_tab_no ( picker, tab));
    }
}

static void
mwb_browser_downloads_progress_cb (MwbDownloadModel *model,
                                   gdouble           progress,
                                   MwbBrowser       *self)
{
  gchar *text;
  MwbTab *tab = mwb_browser_get_downloads_tab (self);

  if (!tab)
    return;

  /* Downloads are active, disable closing */
  mwb_tab_set_can_close (tab, FALSE);

  /* Update the tab title */
  text = g_strdup_printf (_("Downloads - %d%%"), (gint)(progress * 100.0));
  mwb_tab_set_text (tab, text);
  g_free (text);
}

static void
mwb_browser_downloads_complete_cb (MwbDownloadModel *model,
                                   MwbBrowser       *self)
{
  MwbTab *tab = mwb_browser_get_downloads_tab (self);

  if (!tab)
    return;

  /* Update the tab title */
  mwb_tab_set_text (tab, _("Downloads"));

  /* Set the tab as closeable */
  mwb_tab_set_can_close (tab, TRUE);
}

static void
mwb_browser_hide_completed_cb (ClutterTimeline *timeline,
                               MwbBrowser      *self)
{
  MwbBrowserPrivate *priv = self->priv;

  if (clutter_timeline_get_direction (timeline) == CLUTTER_TIMELINE_FORWARD)
    {
      priv->fullscreen = TRUE;
      clutter_actor_set_reactive (CLUTTER_ACTOR (self), TRUE);
      clutter_timeline_set_direction (timeline, CLUTTER_TIMELINE_BACKWARD);
    }
  else
    {
      clutter_timeline_set_direction (timeline, CLUTTER_TIMELINE_FORWARD);
      priv->bar_visible = priv->fullscreen;
    }

  clutter_timeline_rewind (timeline);
}

static void
mwb_browser_init (MwbBrowser *self)
{
  MwbTab *new_tab;
  MwbDownloadModel *model;
  ClutterActor *default_icon;
  MwbRadicalBar *radical_bar;

  MwbBrowserPrivate *priv = self->priv = BROWSER_PRIVATE (self);

  clutter_actor_set_name (CLUTTER_ACTOR (self), "browser");

  priv->home = g_strdup_printf (DEFAULT_HOME);
  priv->private_home = g_strdup_printf (DEFAULT_PRIVATE_HOME);

  priv->history = mhs_history_new ();

  priv->toolbar = mwb_toolbar_new ();
  priv->tab_picker = mwb_tab_picker_new ();
  priv->status_bar = mwb_status_bar_new ();

  mwb_browser_update_title (self, _("Moblin web browser"));

  clutter_actor_set_parent (CLUTTER_ACTOR (priv->toolbar),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->tab_picker),
                            CLUTTER_ACTOR (self));
  clutter_actor_set_parent (CLUTTER_ACTOR (priv->status_bar),
                            CLUTTER_ACTOR (self));

  /* Keep a reference to this default icon for future cloning */
  default_icon = (ClutterActor *)
    nbtk_texture_cache_get_texture (nbtk_texture_cache_get_default (),
                                    DEFAULT_ICON, FALSE);
  clutter_actor_set_parent (default_icon, CLUTTER_ACTOR (self));

  radical_bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
  default_icon = (ClutterActor *)
    nbtk_texture_cache_get_texture (nbtk_texture_cache_get_default (),
                                    DEFAULT_ICON, FALSE);
  mwb_radical_bar_set_default_icon (radical_bar, default_icon);

  /* Add 'new page' tab */
  new_tab = MWB_TAB (mwb_tab_new_full ("", NULL, -1));
  mwb_tab_set_docking (new_tab, TRUE);
  clutter_actor_set_name (CLUTTER_ACTOR (new_tab), "new-tab");
  mwb_tab_set_can_close (new_tab, FALSE);
  mwb_tab_picker_add_tab (MWB_TAB_PICKER (priv->tab_picker), new_tab, -1);

  g_signal_connect (priv->tab_picker, "tab-activated",
                    G_CALLBACK (mwb_browser_tab_activated_cb), self);
  g_signal_connect (priv->tab_picker, "chooser-clicked",
                    G_CALLBACK (mwb_browser_chooser_clicked_cb), self);
  g_signal_connect (priv->tab_picker, "notify::preview-mode",
                    G_CALLBACK (tab_picker_preview_mode_notify), self);
  g_signal_connect (radical_bar, "go",
                    G_CALLBACK (mwb_browser_go_cb), self);
  g_signal_connect (radical_bar, "reload",
                    G_CALLBACK (mwb_browser_reload_cb), self);
  g_signal_connect (radical_bar, "stop",
                    G_CALLBACK (mwb_browser_stop_cb), self);
  g_signal_connect (radical_bar, "pin-button-clicked",
                    G_CALLBACK (mwb_browser_pin_button_clicked_cb), self);
  g_signal_connect_swapped (radical_bar, "notify::loading",
                            G_CALLBACK (update_cursor), self);
  g_signal_connect (priv->toolbar, "travel",
                    G_CALLBACK (mwb_browser_toolbar_travel_cb), self);
  g_signal_connect (priv->toolbar, "settings",
                    G_CALLBACK (mwb_browser_settings_cb), self);
  g_signal_connect (priv->toolbar, "quit",
                    G_CALLBACK (mwb_browser_toolbar_quit_cb), self);

  g_signal_connect (priv->history, "pinned-page",
                    G_CALLBACK (mwb_browser_pinned_page_cb), self);
  g_signal_connect (priv->history, "unpinned-page",
                    G_CALLBACK (mwb_browser_unpinned_page_cb), self);

  /* Create the download manager */
  priv->dlman = g_object_ref_sink (mwb_download_manager_new ());
  model = mwb_download_manager_get_model (MWB_DOWNLOAD_MANAGER (priv->dlman));
  g_signal_connect (priv->dlman, "download-added",
                    G_CALLBACK (mwb_browser_download_added_cb), self);
  g_signal_connect (model, "progress",
                    G_CALLBACK (mwb_browser_downloads_progress_cb), self);
  g_signal_connect (model, "completed",
                    G_CALLBACK (mwb_browser_downloads_complete_cb), self);

  /* Focus the location entry */
  mwb_radical_bar_focus (
    mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar)));

  /* Create timeline/alpha for fullscreening */
  priv->bar_visible = TRUE;
  priv->bar_hide_timeline = clutter_timeline_new (350);
  priv->bar_hide_alpha = clutter_alpha_new_full (priv->bar_hide_timeline,
                                                   CLUTTER_EASE_IN_QUAD);
  g_signal_connect_swapped (priv->bar_hide_timeline, "new-frame",
                            G_CALLBACK (clutter_actor_queue_relayout), self);
  g_signal_connect (priv->bar_hide_timeline, "completed",
                    G_CALLBACK (mwb_browser_hide_completed_cb), self);

  priv->mozembed_cursor = MOZ_HEADLESS_CURSOR_TYPE_LEFT_PTR;
}

ClutterActor *
mwb_browser_new (MwbWindow *window)
{
  return g_object_new (MWB_TYPE_BROWSER, "window", window, NULL);
}

static void
mwb_browser_tab_anim_frame_cb (ClutterTimeline *timeline,
                               gint             frame_num,
                               MwbTab          *tab)
{
  if (!clutter_actor_get_parent (CLUTTER_ACTOR (tab)))
    {
      clutter_timeline_stop (timeline);
      g_object_unref (timeline);
      g_object_unref (tab);

      return;
    }

  mwb_tab_set_width (tab, 200 * clutter_timeline_get_progress (timeline));
}

static void
mwb_browser_tab_anim_complete_cb (ClutterTimeline *timeline,
                                  MwbTab          *tab)
{
  mwb_tab_set_width (tab, 200);
  g_object_unref (tab);
  g_object_unref (timeline);
}

static void
mwb_browser_tab_close_cb (ClutterTimeline *timeline,
                          MwbTab          *tab)
{
  MwbBrowser *browser;
  MwbBrowserTabData *tab_data;

  MwbTabPicker *tab_picker =
    MWB_TAB_PICKER (clutter_actor_get_parent (CLUTTER_ACTOR (tab)));

  tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");
  browser = MWB_BROWSER (tab_data->browser);

  mwb_tab_picker_remove_tab (tab_picker, tab);
  if (tab_data)
    {
      MwbBrowserPrivate *priv = MWB_BROWSER (tab_data->browser)->priv;

      if (MWB_IS_PAGE (tab_data->page))
        {
          ClutterMozEmbed *mozembed =
            mwb_page_get_mozembed (MWB_PAGE (tab_data->page));

          g_free (g_object_get_data (G_OBJECT (mozembed), "request"));

          priv->pages = g_list_remove (priv->pages, tab_data->page);
          if (tab_data->favicon_handler)
            mhs_history_cancel (priv->history,
                                tab_data->favicon_handler);
          if (tab_data->is_pinned_handler)
            mhs_history_cancel (priv->history,
                                tab_data->is_pinned_handler);
        }

      clutter_actor_unparent (tab_data->page);
      g_slice_free (MwbBrowserTabData, tab_data);
    }

  queue_save_session (browser);
}

static void
mwb_browser_get_favicon_cb (MhsHistory   *history,
                            const gchar  *mime_type,
                            const guint8 *data,
                            guint         data_len,
                            const GError *error,
                            gpointer      user_data)
{
  MwbBrowserTabData *tab_data = user_data;

  if (error)
    g_warning ("favicon error: %s\n", error->message);
  else
    {
      MwbTab *tab = mwb_browser_find_tab_for_mozembed (
                      MWB_BROWSER (tab_data->browser),
                      mwb_page_get_mozembed (MWB_PAGE (tab_data->page)));

      if (tab)
        {
          GError *texture_error = NULL;
          CoglHandle cogl_texture = mwb_utils_image_to_texture (data, data_len,
                                                                &texture_error);

          if (cogl_texture != COGL_INVALID_HANDLE)
            {
              ClutterActor *texture = clutter_texture_new ();
              clutter_texture_set_cogl_texture (CLUTTER_TEXTURE (texture),
                                                cogl_texture);
              mwb_tab_set_icon (tab, texture);
              cogl_handle_unref (cogl_texture);
            }
          else
            {
              g_warning ("favicon error: %s", texture_error->message);
              g_error_free (texture_error);
            }
        }
    }

  tab_data->favicon_handler = 0;
}

static void
mwb_browser_start_loading_favicon (MwbBrowser *browser,
                                   MwbBrowserTabData *tab_data)
{
  MwbBrowserPrivate *priv = browser->priv;

  mwb_browser_stop_loading_favicon (browser, tab_data);

  if (priv->history && tab_data && MWB_IS_PAGE (tab_data->page))
    {
      const gchar *location =
        clutter_mozembed_get_location (
          mwb_page_get_mozembed (MWB_PAGE (tab_data->page)));

      if (location)
        tab_data->favicon_handler =
          mhs_history_get_favicon (priv->history,
                                   location,
                                   TRUE,
                                   mwb_browser_get_favicon_cb,
                                   tab_data,
                                   NULL);
    }
}

static void
mwb_browser_stop_loading_favicon (MwbBrowser        *browser,
                                  MwbBrowserTabData *tab_data)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->history && tab_data->favicon_handler)
    {
      mhs_history_cancel (priv->history, tab_data->favicon_handler);
      tab_data->favicon_handler = 0;
    }
}

static void
mwb_browser_tab_closed_cb (MwbTab *tab, MwbBrowser *browser)
{
  gint current_tab;
  MwbTabPicker *tab_picker;
  MwbBrowserTabData *tab_data;
  ClutterTimeline *timeline;
  MwbBrowserPrivate *priv = browser->priv;

  tab_picker = MWB_TAB_PICKER (priv->tab_picker);

  /* Disconnect the ClutterMozEmbed */
  tab_data = g_object_get_data (G_OBJECT (tab), "browser-tab-data");
  if (MWB_IS_PAGE (tab_data->page))
    mwb_browser_embed_disconnect (browser, MWB_PAGE (tab_data->page));

  /* Pick prior/next tab */
  current_tab = mwb_tab_picker_get_current_tab (tab_picker);
  if (mwb_tab_picker_get_tab (tab_picker, current_tab) == tab)
    {
      if (current_tab > 0)
        mwb_tab_picker_set_current_tab (tab_picker, current_tab - 1);
      else
        mwb_tab_picker_set_current_tab (tab_picker, current_tab + 1);
    }
  else if (mwb_tab_picker_get_tab_no (tab_picker, tab) < priv->current_tab)
    priv->current_tab --;

  /* Run tab-closing animation */
  clutter_actor_set_reactive (CLUTTER_ACTOR (tab), FALSE);
  timeline = clutter_timeline_new (150);
  clutter_timeline_set_direction (timeline, CLUTTER_TIMELINE_BACKWARD);
  g_signal_connect (timeline, "new-frame",
                    G_CALLBACK (mwb_browser_tab_anim_frame_cb), tab);
  g_signal_connect (timeline, "completed",
                    G_CALLBACK (mwb_browser_tab_close_cb), tab);
  clutter_timeline_start (timeline);
}

gboolean
mwb_browser_button_press_cb (ClutterActor       *actor,
                             ClutterButtonEvent *event,
                             MwbBrowser         *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (priv->current_link)
    {
      g_free (priv->current_link);
      priv->current_link = NULL;
    }

  /* on ctrl-click or middle-click save link for opening a new tab */
  if ((event->button == 1 && (event->modifier_state & CLUTTER_CONTROL_MASK)) ||
      event->button == 2)
    {
      const gchar *text;
      text = mwb_status_bar_get_text (MWB_STATUS_BAR (priv->status_bar));
      if (text && text[0] != '\0')
        priv->current_link = g_strdup (text);
    }

  return FALSE;
}

gboolean
mwb_browser_button_release_cb (ClutterActor       *actor,
                               ClutterButtonEvent *event,
                               MwbBrowser         *browser)
{
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->current_link)
    return FALSE;

  if ((event->button == 1 && (event->modifier_state & CLUTTER_CONTROL_MASK)) ||
      event->button == 2)
    {
      const gchar *text;
      text = mwb_status_bar_get_text (MWB_STATUS_BAR (priv->status_bar));
      if (!g_strcmp0 (text, priv->current_link))
        {
          MwbBrowserOpenFlags flags = MWB_BROWSER_OPEN_NEW_TAB;

          if (priv->current_page && MWB_IS_PAGE (priv->current_page))
            {
              ClutterMozEmbed *mozembed =
                mwb_page_get_mozembed (MWB_PAGE (priv->current_page));

              if (mozembed && clutter_mozembed_get_private (mozembed))
                flags |= MWB_BROWSER_OPEN_PRIVATE;
            }

          mwb_browser_open (browser, text, flags);
        }
    }

  return FALSE;
}

MwbTab *
mwb_browser_add_tab (MwbBrowser     *browser,
                     const gchar    *title,
                     ClutterActor    *actor,
                     gboolean         new_tab,
                     gboolean         activate)
{
  MwbTab *tab;
  ClutterActor *clone;
  MwbTabPicker *picker;
  gint tab_no, no_tabs;
  MwbBrowserTabData *tab_data;
  MwbBrowserPrivate *priv = browser->priv;

  picker = MWB_TAB_PICKER (priv->tab_picker);

  /* Create tab */
  tab_no = priv->current_tab;
  tab = new_tab ? NULL : mwb_tab_picker_get_tab (picker, tab_no);

  /* TODO: Make this width stylable or an algorithmic value based on
   * screen/window/font size
   */
  if (!tab)
    {
      ClutterActor *default_icon;
      ClutterTimeline *timeline;

      tab = MWB_TAB (mwb_tab_new_full (title ? title : _("Untitled"), NULL, 1));
      default_icon = (ClutterActor *)
        nbtk_texture_cache_get_texture (nbtk_texture_cache_get_default (),
                                        DEFAULT_ICON, FALSE);
      mwb_tab_set_default_icon (tab, default_icon);
      g_signal_connect (tab, "notify::icon",
                        G_CALLBACK (mwb_browser_tab_icon_notify_cb), browser);

      new_tab = TRUE;

      /* Create an animation that grows the tab to full size */
      g_object_ref (tab);
      timeline = clutter_timeline_new (150);
      g_signal_connect (timeline, "new-frame",
                        G_CALLBACK (mwb_browser_tab_anim_frame_cb), tab);
      g_signal_connect (timeline, "completed",
                        G_CALLBACK (mwb_browser_tab_anim_complete_cb), tab);
      clutter_timeline_start (timeline);
    }

  g_signal_connect (tab, "closed",
                    G_CALLBACK (mwb_browser_tab_closed_cb), browser);

  if (new_tab)
    {
      /* Don't place the tab after the 'new tab' tab */
      no_tabs = mwb_tab_picker_get_n_tabs (picker);
      if (tab_no != no_tabs - 1)
        tab_no ++;

      mwb_tab_picker_add_tab (picker, tab, tab_no);
    }

  clutter_actor_set_reactive (actor, FALSE);
  clutter_actor_set_parent (actor, CLUTTER_ACTOR (browser));
  priv->pages = g_list_prepend (priv->pages, actor);

  clone = mwb_aspect_clone_new (actor);
  mwb_tab_set_preview_actor (tab, clone);

  tab_data = g_slice_new0 (MwbBrowserTabData);
  tab_data->browser = CLUTTER_ACTOR (browser);
  tab_data->page = actor;
  g_object_set_data (G_OBJECT (tab), "browser-tab-data", tab_data);

  if (MWB_IS_PAGE (actor))
    {
      MwbPage *page = MWB_PAGE (actor);
      ClutterMozEmbed *mozembed = mwb_page_get_mozembed (page);

      mwb_browser_embed_connect (browser, page);

      g_signal_connect (mozembed, "button-press-event",
                        G_CALLBACK (mwb_browser_button_press_cb),
                        browser);
      g_signal_connect (mozembed, "button-release-event",
                        G_CALLBACK (mwb_browser_button_release_cb),
                        browser);

      if (clutter_mozembed_get_private (mozembed))
        mwb_tab_set_private (tab, TRUE);
    }

  if (activate)
    {
      MwbRadicalBar *bar;

      mwb_tab_picker_set_current_tab (picker, tab_no);
      bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
      mwb_radical_bar_focus (bar);
    }

  return tab;
}

static gboolean
mwb_browser_does_url_match (const gchar *url, const gchar *source)
{
  /* returns TRUE if url matches source, or would match if it was patched up
     with http:// prefix and/or trailing slash */

  gboolean matches = FALSE;
  gchar *ptr = strstr (source, url);

  if (ptr)
    {
      guint len = strlen (url);
      guint srclen = strlen (source);

      switch (srclen - len)
        {
        case 0:
          /* strings are equal */
          matches = TRUE;
          break;

        case 1:
          /* check for trailing slash */
          if ((ptr == source) &&
              (source[srclen - 1] == '/'))
            matches = TRUE;
          break;

        case 7:
          /* check for http:// prefix */
          if ((ptr == source + 7) &&
              (!strncmp (source, "http://", 7)))
            matches = TRUE;
          break;

        case 8:
          /* check for both prefix and trailing slash */
          if ((ptr == source + 7) &&
              (!strncmp (source, "http://", 7)) &&
              (source[srclen - 1] == '/'))
            matches = TRUE;
          break;

        default:
          break;
        }
    }

  return matches;
}

void
mwb_browser_open (MwbBrowser          *browser,
                  const gchar         *url,
                  MwbBrowserOpenFlags  open_flags)
{
#ifdef MWB_SINGLE_PROCESS
  static ClutterMozEmbed *parent = NULL;
#endif
  ClutterActor *page;
  ClutterMozEmbed *mozembed;
  MwbBrowserPrivate *priv = browser->priv;

  /* Use an existing tab if there is one */
  if (url && (open_flags & MWB_BROWSER_OPEN_USE_EXISTING))
    {
      gint page_no;
      GList *p, *pages;

      /* Check to see if the url is already open first */
      page_no = 0;
      pages = mwb_browser_get_pages (browser);
      for (p = pages; p; p = p->next)
        {
          const gchar *current_url;
          ClutterMozEmbed *mozembed = p->data;

          current_url = (gchar *)g_object_get_data (G_OBJECT (mozembed),
                                                    "request");
          if (!current_url)
            current_url = clutter_mozembed_get_location (mozembed);

          if (current_url && (clutter_mozembed_get_private (mozembed) ==
                              !!(open_flags & MWB_BROWSER_OPEN_PRIVATE)))
            {
              if (mwb_browser_does_url_match (url, current_url))
                {
                  MwbTabPicker *tab_picker;

                  tab_picker = mwb_browser_get_tab_picker (browser);
                  mwb_tab_picker_set_current_tab (tab_picker, page_no);
                  g_list_free (pages);
                  return;
                }
            }

          page_no ++;
        }
      g_list_free (pages);
    }

  /* Create window */
#ifdef MWB_SINGLE_PROCESS
  page = CLUTTER_ACTOR (mwb_page_new_with_parent (parent));
  if (!parent)
    parent = mwb_page_get_mozembed (MWB_PAGE (page));
#else
  page = CLUTTER_ACTOR (mwb_page_new ((open_flags & MWB_BROWSER_OPEN_PRIVATE) ?
                                      TRUE : FALSE));
#endif
  mwb_page_set_download_manager (MWB_PAGE (page),
                                 MWB_DOWNLOAD_MANAGER (priv->dlman));

  clutter_actor_hide (page);

  mwb_browser_add_tab (browser, NULL, page,
                       !!(open_flags & MWB_BROWSER_OPEN_NEW_TAB),
                       !!(open_flags & MWB_BROWSER_OPEN_ACTIVATE));

  mozembed = mwb_page_get_mozembed (MWB_PAGE (page));
  mwb_browser_wrapped_open (mozembed, url ? url : priv->home);

  if ((open_flags & MWB_BROWSER_OPEN_USER_INITIATED))
    {
      MwbRadicalBar *bar;

      priv->user_initiated_url = TRUE;

      /* Populate radical bar immediately with user-initiated URL */
      bar = mwb_toolbar_get_radical_bar (MWB_TOOLBAR (priv->toolbar));
      mwb_radical_bar_set_text (bar, url);
    }

  queue_save_session (browser);
}

MwbToolbar *
mwb_browser_get_toolbar (MwbBrowser *browser)
{
  return MWB_TOOLBAR (browser->priv->toolbar);
}

MwbTabPicker *
mwb_browser_get_tab_picker (MwbBrowser *browser)
{
  return MWB_TAB_PICKER (browser->priv->tab_picker);
}

GList *
mwb_browser_get_pages (MwbBrowser *browser)
{
  GList *tabs, *t, *pages;

  MwbBrowserPrivate *priv = browser->priv;

  pages = NULL;
  tabs = mwb_tab_picker_get_tabs (MWB_TAB_PICKER (priv->tab_picker));
  for (t = tabs; t; t = t->next)
    {
      MwbBrowserTabData *tab_data
        = g_object_get_data (G_OBJECT (t->data), "browser-tab-data");

      if (!tab_data || !MWB_IS_PAGE (tab_data->page))
        continue;

      pages = g_list_append (pages,
                             mwb_page_get_mozembed (MWB_PAGE (tab_data->page)));
    }
  g_list_free (tabs);

  return pages;
}

MwbStatusBar *
mwb_browser_get_status_bar (MwbBrowser *browser)
{
  return MWB_STATUS_BAR (browser->priv->status_bar);
}

void
mwb_browser_activate_private (MwbBrowser *browser)
{
  MwbTab *tab;
  MwbPage *page;
  MwbBrowserPrivate *priv = browser->priv;

  if (!priv->current_page || !MWB_IS_PAGE (priv->current_page))
    return;

  page = MWB_PAGE (priv->current_page);

  mwb_browser_embed_disconnect (browser, page);

  mwb_page_set_mozembed (page, MWB_MOZEMBED (mwb_mozembed_new (TRUE)));
  mwb_browser_embed_connect (browser, page);

  if (priv->private_home)
    mwb_browser_wrapped_open (mwb_page_get_mozembed (page), priv->private_home);

  tab = mwb_browser_find_tab_for_mozembed (browser,
                                           mwb_page_get_mozembed (page));
  mwb_tab_set_private (tab, TRUE);
}

void
mwb_browser_purge_session_history (MwbBrowser *browser)
{
  GList *pages, *p;

  pages = mwb_browser_get_pages (browser);
  for (p = pages; p; p = p->next)
    clutter_mozembed_purge_session_history (CLUTTER_MOZEMBED (p->data));
  g_list_free (pages);
}
