/*
 * Copyright (C) 2008  Intel Corporation
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License, version 2.1, as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

/**
 * @file UserSession.c
 *
 * Implementation of UserSession library
 */

#include <glib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib.h>
#include "UserSession.h"
#include <stdio.h>

#define CK_NAME "org.freedesktop.ConsoleKit"
#define CK_MANAGER_PATH "/org/freedesktop/ConsoleKit/Manager"
#define CK_MANAGER_INTERFACE "org.freedesktop.ConsoleKit.Manager"
#define CK_SESSION_INTERFACE "org.freedesktop.ConsoleKit.Session"

static DBusGConnection *connection = NULL;
static DBusGProxy *proxy_manager = NULL;
static DBusGProxy *proxy_session = NULL;
static char *ssid = NULL;	/* current session id */

int InitializeUserSessionStatus(void)
{
	GError *error = NULL;
	gboolean result;

	g_type_init();

	connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
	if (NULL == connection) {
		g_error_free(error);
		return USS_Uninitialized;
	}

	proxy_manager = dbus_g_proxy_new_for_name(connection, CK_NAME,
		CK_MANAGER_PATH, CK_MANAGER_INTERFACE);
	if (NULL == proxy_manager) {
		return USS_Uninitialized;
	}

	result = dbus_g_proxy_call(proxy_manager, "GetCurrentSession", &error,
		G_TYPE_INVALID, DBUS_TYPE_G_OBJECT_PATH, &ssid, G_TYPE_INVALID);
	if (!result) {
		g_warning("%s\n");
		g_error_free(error);
		return USS_Uninitialized;
	}

	proxy_session = dbus_g_proxy_new_for_name(connection, CK_NAME, ssid,
		CK_SESSION_INTERFACE);
	if (NULL == proxy_session) {
		g_object_unref(proxy_manager);
		return USS_Uninitialized;
	}

	return 0;
}

void UninitializeUserSessionStatus(void)
{
	if (NULL != proxy_manager) {
		g_object_unref(proxy_manager);
		proxy_manager = NULL;
	}

	if (NULL != proxy_session) {
		g_object_unref(proxy_session);
		proxy_session = NULL;
	}
}

UserSessionStatus GetUserSessionStatus(void)
{
	GError *error = NULL;
	gboolean result;
	gboolean active;
	gboolean local;

	if (NULL == proxy_session) {
		return USS_Uninitialized;
	}

	result = dbus_g_proxy_call(proxy_session, "IsActive", &error,
		G_TYPE_INVALID, G_TYPE_BOOLEAN, &active, G_TYPE_INVALID);
	if (!result) {
		return USS_UnexpectedError;
	}

	result = dbus_g_proxy_call(proxy_session, "IsLocal", &error,
		G_TYPE_INVALID, G_TYPE_BOOLEAN, &local, G_TYPE_INVALID);
	if (!result) {
		return USS_UnexpectedError;
	}

	if (active) {
		if (local) {
			return USS_ActiveSession;
		} else {
			return USS_ActiveRemoteSession;
		}
	} else {
		return USS_InactiveSession;
	}
}
